/*
**  PROJECT
**      SortDNS
**
**  DESCRIPTION
**      This filter sorts lines where the first field is treated
**      as a DNS name.  The resulting sort is from the top level
**      domain down.
**
**  DEVELOPER
**      Scott C. Karlin
**
**  HISTORY
**      07 Jul 2003  sck  Initial Version
**      22 Sep 2004  sck  Added usage, RPM-ized
**
**  CVS ID
**      $Id: sortDNS.c,v 1.1.1.1 2004/09/22 21:13:12 sck-SortDNS Exp $
*/

#define  _GNU_SOURCE      /* get GNU extensions (getline, program_...) */

#include <assert.h>       /* assert */
#include <ctype.h>        /* isalnum, tolower */
#include <errno.h>        /* program_... */
#include <getopt.h>       /* getopt_long, no_argument, required_argument */
#include <stdio.h>        /* getline, printf */
#include <stdlib.h>       /* calloc, exit, malloc, qsort, EXIT_* */
#include <string.h>       /* strchr, strcmp, strdup, strlen */

#define MAXLEVELS      20

/****************************************************************************/

static int Reverse_flag = 0;

/****************************************************************************/
/*
**  Additional characters (beyond isalnum) that we accept in DNS names
**  for sorting purposes.  Technically, the "-" is the only non alnum
**  character allowed.  However, "_" has been seen in the wild and the
**  others are often in regular expressions that we may want to sort.
*/

static const char *charset = "-_*[]";

/****************************************************************************/

typedef struct Line Line_T;

struct Line
   {
      char   *line;             /* original line from getline   */
      char   *lower;            /* above converted to lowercase, and then split */
      char   *dns[MAXLEVELS];   /* pointers into lower, top-level domain first */
      Line_T *next;             /* singly-linked list */
   };



static int dnscomp(const void *a, const void *b)
{
   const Line_T *p1 = *((const Line_T **) a);
   const Line_T *p2 = *((const Line_T **) b);
   char *s1;
   char *s2;
   int cmp;
   int i;
   int retval;

   retval = 0;
   for(i = 0; i < MAXLEVELS; i += 1)
      {
         s1 = p1->dns[i];
         s2 = p2->dns[i];

         if(s1 == NULL)
            {
               retval = (s2 == NULL) ? 0 : -1;
               break;
            }

         if(s2 == NULL)
            {
               retval = 1;
               break;
            }

         cmp = strcmp(s1, s2);
         if(cmp != 0)
            {
               retval = (cmp < 0) ? -1 : 1;
               break;
            }
      }
   return Reverse_flag ? -retval : retval;
}

/****************************************************************************/
/*
**  Usage Message and Command Line Parameter Processing
*/

static const char * const UsageTemplate =
   "usage: %s [options]\n"
   "Read standard input, sort by DNS name, write to standard output.\n"
   "   where:\n"
   "      -h/--help\n"
   "             Display this help message and exit.\n"
   "\n"
   "      -r/--reverse\n"
   "             Reverse the order of the sort.\n"
   ;


/*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  */

static void usage(int code) __attribute__ ((noreturn));

static void usage(int code)
{
   FILE *fp;

   fp = (code == EXIT_SUCCESS) ? stdout : stderr;

   fprintf(fp, UsageTemplate, program_invocation_short_name);

   exit(code);
}

/*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  */

static const struct option OptionLong[] =
   {
         { "help",     no_argument,       NULL, 'h' },
         { "reverse",  no_argument,       NULL, 'r' },
         { NULL,       0,                 NULL, 0   },
   };

static const char * const OptionShort = "hr";

/*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  */

static void parse_options(int argc, char *argv[],
                          int *ptr_reverse)
{
   int   count_reverse;
   int   i;
   int   opt;
   int   opt_reverse;
   int   opt_help;
   int   parse_error;
   /*
   **  Check parameters
   */
   assert(ptr_reverse != NULL);

   count_reverse = 0;
   parse_error   = 0;
   opt_reverse   = 0;
   opt_help      = 0;

   /*
   **  Shorten the error message for unrecognized options
   */
   argv[0] = program_invocation_short_name;

   while((opt = getopt_long(argc, argv, OptionShort, OptionLong, NULL)) != -1)
      {
         switch(opt)
            {
               case 'r':                 /* -r/--reverse */
                  count_reverse += 1;
                  opt_reverse = 1;
                  break;

               case 'h':                 /* -h/--help */
                  opt_help = 1;
                  break;

               case '?':                 /* unknown option */
                  /* (error message has already been displayed) */
                  parse_error = 1;
                  break;

               default:
                  assert(!"Internal Error");
            }
      }

   if(optind < argc)
      {
         fprintf(stderr, "%s: unrecognized parameters: \"", program_invocation_short_name);
         for(i = optind; i < argc; i += 1)
            {
               fprintf(stderr, "%s%s", argv[i], (i == argc - 1) ? "\"\n" : " ");
            }
         parse_error = 1;
      }

   if(count_reverse > 1)
      {
         fprintf(stderr, "%s: too many -e/--reverse options given\n",
                         program_invocation_short_name);
         parse_error = 1;
      }

   if(parse_error)
      {
         usage(EXIT_FAILURE);
      }

   if(opt_help)
      {
         usage(EXIT_SUCCESS);
      }

   *ptr_reverse = opt_reverse;
}


/****************************************************************************/
/* FIXME: add usage information */

int main(int argc, char *argv[])
{
   Line_T   **ptrs;
   Line_T   *curr;
   Line_T   *head;
   Line_T   *tail;
   char     *dns[MAXLEVELS];
   char     *line;
   char     *p;
   int       numl;
   int       i;
   int       s;
   int       v;
   size_t    n;
   ssize_t   len;

   parse_options(argc, argv, &Reverse_flag);

   line = NULL;
   n    = 0;

   head = NULL;
   tail = NULL;
   numl = 0;

   while((len = getline(&line, &n, stdin)) != -1)
      {
         curr = calloc(1, sizeof(*curr));
         assert(curr != NULL);

         curr->line = strdup(line);
         assert(curr->line != NULL);

         curr->lower = strdup(line);
         assert(curr->lower != NULL);

         len = strlen(line);

         for(i = 0; i < len; i += 1)
            {
               curr->lower[i] = tolower(curr->lower[i]);
            }

         /* create pointers to domain elements */
         p = curr->lower;
         v = 0;
         s = 0;
         while(1)
            {
               if(s == 0)
                  {
                     if(isalnum(*p) || (strchr(charset, *p) != NULL))
                        {
                           dns[v] = p;
                           v += 1;
                           p += 1;
                           s = 1;
                        }
                     else
                        {
                           break;
                        }
                  }
               else
                  {
                     if(isalnum(*p) || (strchr(charset, *p) != NULL))
                        {
                           p += 1;
                        }
                     else if((*p == '.') && (v < MAXLEVELS))
                        {
                           *p = '\0';
                           p += 1;
                           s = 0;
                        }
                     else
                        {
                           *p = '\0';
                           break;
                        }

                  }
            }
         /*
         **  Copy the list in reverse order
         */
         for(i = 0; i < v; i += 1)
            {
               curr->dns[i] = dns[v - i - 1];
            }
         /*
         **  Link
         */
         if(head == NULL)
            {
               head = curr;
               tail = curr;
            }
         else
            {
               tail->next = curr;
               tail       = curr;
            }
         numl += 1;
      }
   /* create array of pointers */
   ptrs = malloc(numl * sizeof(*ptrs));
   assert(ptrs != NULL);

   curr = head;
   for(i = 0; i < numl; i += 1)
      {
         ptrs[i] = curr;
         curr = curr->next;
      }

   /* sort array */
   qsort(ptrs, numl, sizeof(*ptrs), dnscomp);

   /* print array */
   for(i = 0; i < numl; i += 1)
      {
         printf("%s", ptrs[i]->line);
      }

   exit(EXIT_SUCCESS);
}

/****************************************************************************/
