/*
**  DESCRIPTION
**      Daemonize routine
**
**  DEVELOPER
**      Scott C. Karlin   <scott@cs.princeton.edu>
**
**  HISTORY
**      03 Jul 2003  sck  Initial Version
**      21 Sep 2004  sck  RPM-ized
**
**  CVS ID
**      $Id$
*/

#include <errno.h>        /* errno */
#include <fcntl.h>        /* open */
#include <stdio.h>        /* snprintf */
#include <stdlib.h>       /* exit, EXIT_* */
#include <unistd.h>       /* close, dup, getdtablesize, fork, getppid, setsid */
#include <sys/file.h>     /* flock */
#include <sys/stat.h>     /* umask */
#include <sys/types.h>    /* pid_t */

/****************************************************************************/

#define PIDSTRLEN        20

/****************************************************************************/

void daemonize(char *dir, char *lockfile)
{
   char  pidstr[PIDSTRLEN];
   int   fd;
   int   len;
   int   retval;
   pid_t pid;
/*
**  If we are already a daemon (parent is pid=1 -- the init process),
**  then there's nothing to do.
*/
   if(getppid() == 1)
      {
         return;
      }
/*
**  Clear the file creation mask so that permissions are not limited.
*/
   umask(0);
/*
**  Fork
**     Parent exits so that child becomes detatched from the shell.
**     Since the child is detatched from the shell, it is now in the
**     background and is not a process group leader.
*/
   pid = fork();
   if(pid < 0)
      {
         fprintf(stderr, "libdaemon: fork error");
         exit(EXIT_FAILURE);
      }
   if(pid > 0)
      {
         exit(EXIT_SUCCESS);    /* parent exits */
      }
/*
**  (At this point, we are the child process and the parent has exited.)
**
**  The setsid() call does three things:
**    (1) this process becomes the session leader of a new session
**    (2) this process becomes process group leader of a new process group
**    (3) any controlling terminal is released
*/
   setsid();
/*
**  Close all open file descriptors
*/
   for(fd = getdtablesize() - 1; fd >= 0; fd -= 1)
      {
         do
            {
               retval = close(fd);
            }
         while((retval == -1) && (errno == EINTR));
      }
/*
**  Open stdin (0), stdout (1), and stderr (2) as /dev/null
*/
   fd = open("/dev/null", O_RDWR);         /* stdin */
   if(fd < 0)
      {
         _exit(EXIT_FAILURE);
      }
   dup(fd);                                /* stdout */
   dup(fd);                                /* stderr */
/*
**  Change directory to root in case the daemon's current directory
**  is a mounted file system.
*/
   if(chdir(dir) < 0) 
      {
         _exit(EXIT_FAILURE);
      }
/*
**  Lock file processing (optional, only if lockfile is not NULL)
*/
   if(lockfile != NULL)
      {
         /*
         **  Attempt to get a lock.
         **  If fails, then another instance is running
         */
         fd = open(lockfile, O_RDWR | O_CREAT, 0640);
         if(fd < 0)
            {
               _exit(EXIT_FAILURE);
            }
         if(lockf(fd, F_TLOCK, 0) < 0)
            {
               _exit(EXIT_SUCCESS);
            }
         /*
         **  Record this process's pid in the file
         **  Note that we do not close the lock file.
         */
         len = snprintf(pidstr, PIDSTRLEN, "%d\n", getpid());
         if((len <= 0) || (len > (PIDSTRLEN - 1)))
            {
               _exit(EXIT_FAILURE);
            }
         write(fd, pidstr, len);
      }
/*
**  Success.  The process is now a bona fide daemon.
*/
}

/****************************************************************************/
