/*
**  PROJECT
**      Comma Separated Value Library (Libcsv)
**
**  DESCRIPTION
**      Test file for the CSV library
**
**  DEVELOPER
**      Scott C. Karlin
**
**  HISTORY
**      23 May 2003  sck  Initial Version
**
**  CVS ID
**      $Id: main.c,v 1.2 2003/05/23 17:08:49 scott Exp $
*/

#define  _GNU_SOURCE      /* get GNU extensions (getline, program_...) */

#include <assert.h>       /* assert */
#include <ctype.h>        /* isdigit, isspace */
#include <errno.h>        /* errno, program_... */
#include <stddef.h>       /* NULL */
#include <stdio.h>        /* fprintf */
#include <stdlib.h>       /* EXIT_FAILURE, EXIT_SUCCESS */
#include <string.h>       /* strcmp */

#include <csv/csv.h>      /* CSV: csv_* */

/****************************************************************************/

//#define MAIN_DEBUG

#ifdef MAIN_DEBUG
#define dprintf(format, args...) fprintf(stderr, format, args)
#else
#define dprintf(format, args...)
#endif

/****************************************************************************/

static int hexof(int c)
{
   const char *s = "0123456789ABCDEF0123456789abcdef";
   char *p;

   p = strchr(s, c);

   return (p == NULL) ? 0 : ((p - s) & 0xF);
}

/****************************************************************************/

static void encode(void)
{
   char    *dp;
   char    *line;
   char    *sp;
   char    *csv;
   int      first;
   int      neednl;
   size_t   size;
   ssize_t  len;

   dprintf("%s:\n", __FUNCTION__);
   first  = 1;
   neednl = 0;
   line   = NULL;
   size   = 0;
   while((len = getline(&line, &size, stdin)) > 0)
      {
         assert(line[len - 1] == '\n');
         line[len - 1] = '\0';
         len -= 1;
         dprintf("%s (%d) : len=%2d  line=\"%s\"\n", __FUNCTION__, __LINE__, len, line);

         if(len == 0)
            {
               if(neednl)
                  {
                     printf("\n");
                     first  = 1;
                     neednl = 0;
                  }
               else
                  {
                     continue;
                  }
            }
         else
            {
               sp = line;
               dp = line;
               while(isxdigit(sp[0]) && isxdigit(sp[1]))
                  {
                     *dp = (hexof(sp[0]) << 4) | hexof(sp[1]);
                     dp += 1;
                     sp += 2;
                  }
               *dp = '\0';
               if(!first)
                  {
                     printf(",");
                  }
               first = 0;
               csv = csv_StringEncode(line);
               printf("%s", csv);
               free(csv);
               csv = NULL;
               neednl = 1;
            }
      }
   if(neednl)
      {
         printf("\n");
      }
   free(line);
   line = NULL;
}

/****************************************************************************/

static void decode(void)
{
   csv_Split_t *csv;
   char    *line;
   char    *p;
   int      i;
   int      retval;
   size_t   size;
   ssize_t  len;

   dprintf("%s:\n", __FUNCTION__);
   line   = NULL;
   size   = 0;
   while((len = getline(&line, &size, stdin)) > 0)
      {
         assert(line[len - 1] == '\n');
         line[len - 1] = '\0';
         len -= 1;
         dprintf("%s (%d) : len=%2d  line=\"%s\"\n", __FUNCTION__, __LINE__, len, line);
         retval = csv_SplitNew(line, &csv);
         if(retval == 1)
            {
               fprintf(stderr, "parse error! -- skipping\n");
               continue;
            }
         assert(retval == 0);
         dprintf("%s (%d) : csv->num=%2d  csv->field[0]=\"%s\"\n", __FUNCTION__, __LINE__, csv->num, csv->field[0]);
         for(i = 0; i < csv->num; i += 1)
            {
               for(p = csv->field[i]; *p != '\0'; p += 1)
                  {
                     printf("%02X", ((int) *p) & 0x00FF);
                  }
               printf(" ");
               for(p = csv->field[i]; *p != '\0'; p += 1)
                  {
                     if(isgraph(*p))
                        {
                           printf("%c", *p);
                        }
                     else
                        {
                           printf(".");
                        }
                  }
               printf("\n");
            }
         printf("\n");
         csv_SplitFree(&csv);
      }
   free(line);
   line = NULL;
}

/****************************************************************************/

static void usage(void)
{
   fprintf(stderr,
           "usage: %s [-d|-e]\n"
           "\n"
           "where:\n"
           "   -d  -  decode a CSV file to a machine friendly format (default)\n"
           "   -e  -  encode the machine friendly format to CSV\n"
           "This program is a filter.\n"
           "\n",
           program_invocation_short_name);
}

/****************************************************************************/
/*
**  The main routine
*/

int main(int argc, char **argv)
{
   if((argc == 1) || ((argc == 2) && (strcmp(argv[1], "-d") == 0)))
      {
         decode();
      }
   else if((argc == 2) && (strcmp(argv[1], "-e") == 0))
      {
         encode();
      }
   else
      {
         usage();
         return(EXIT_FAILURE);
      }

   return EXIT_SUCCESS;
}

/****************************************************************************/
