/*
**  PROJECT
**      CSV Library
**
**  DESCRIPTION
**      Interface to Libcsv, a simple C library for manipulating
**      CSV (Comma Separated Value) files.
**
**  DEVELOPER
**      Scott C. Karlin
**
**  HISTORY
**      24 Apr 2003  sck  Initial Version
**
**  CVS ID
**      $Id: csv.h,v 1.4 2003/05/23 17:08:49 scott Exp $
*/

#ifndef CSV_H
#define CSV_H

/****************************************************************************/
/*
**  DESCRIPTION
**      The <csv_Split_t> type holds a broken down (split up) CSV line.
**      The csv_SplitNew allocates and fills one of these types by
**      parsing a CSV string of the form:
**
**         CSV_line   = sp* field [ sp* delim sp* field ]* sp*
**
**      where
**
**         field            = non_quoted_value | quoted_value
**         delim            = ","
**         non_quoted_value = non-empty string that contains only
**                            printable ASCII characters 0x20-0x7E
**                            (except double-quotes, commas, or
**                            backslashes) and does not start or end
**                            with a space.
**         quoted_value     = same format as a string constant in C.
**
**      Example
**
**         12,  34,  "Hello, World!\n", "",1,2,3
**
**      The <field> component of csv_Split_t is a vector of pointers
**      to the split strings.
**
**      The <field> vector has an additional NULL pointer to indicate
**      the end of the list.  In addition, <num> is the number of valid
**      (not counting the terminating NULL pointer) entries in <field>.
**
**      Special Cases
**         * A line with a # in the first position is treated as a
**           blank line.
**         * A \0 in a quoted_value will be ignored.  (So as not to
**           generate a runt string.)
*/

typedef struct csv_split
  {
      char **field;
      int    num;
      void  *priv;       /* private area: holds strings and vectors */
  } csv_Split_t;


/****************************************************************************/

extern int csv_SplitNew(const char *csv, csv_Split_t **pscsv);

/*
**  DESCRIPTION
**      Parse the line <csv> and put the resulting pieces into a newly
**      allocated csv_Split_t.  The address of the newly allocated structure
**      is returned through <pscsv>.  When it is no longer needed, free the
**      structure with a call to csv_SplitFree.
**
**      Parsing stops when a '\0', '\r', or '\n' is encountered in <csv>.
**      A '\r' and/or '\n' may only occur at the end of a string.
**      The tab character is the only allowed control character in <csv>.
**
**  PARAMETERS
**      csv    -  The string to split.
**      pscsv  -  The address of a pointer to the allocated csv_Split_t.
**                (The address of the newly allocated csv_Split_t is
**                stored here.)
**
**  RETURN VALUE
**       0  -  Success
**      -1  -  Failure: out of memory
**       1  -  Failure: CSV parse error
*/
/****************************************************************************/

extern void csv_SplitFree(csv_Split_t **pscsv);

/*
**  DESCRIPTION
**      Free the memory associated with <**pscsv>.  The address of the
**      pointer is passed to this routine so that it can set the pointer
**      to NULL after freeing the memory.
**
**  PARAMETERS
**      pscsv  -  The address of a pointer to a previously allocated
**                csv_Split_t.  This function sets <*pscsv> to NULL
**                so that the freed memory won't be accessed through
**                this pointer.
*/
/****************************************************************************/

extern char *csv_StringEncode(const char *s);

/*
**  DESCRIPTION
**      Allocate an "encoded" version of <s>.  If <s> does not contain
**      any special characters, the return value is a copy of <s>.
**      The special characters are double-quote, backslash, space, and
**      any control character.  Note that the return value must be freed
**      using free().
**
**  PARAMETERS
**      s  -  The string to encode.
**
**  RETURN VALUE
**      Success  -  a pointer to a newly allocated encoded string.
**      Failure  -  NULL
*/
/****************************************************************************/

#endif  /* CSV_H */
