#!/usr/bin/env python

#-----------------------------------------------------------------------
# conditions.py
# Author: Bob Dondero
#-----------------------------------------------------------------------

import threading

#-----------------------------------------------------------------------

class BankAcct:

    def __init__(self):
        self._balance = 0
        self._lock = threading.RLock()
        self._condition = threading.Condition(self._lock)

    def get_balance(self):
        self._lock.acquire()
        try:
            return self._balance
        finally:
            self._lock.release()

    def deposit(self, amount):
        self._condition.acquire()
        self._balance += amount
        print(self._balance)
        self._condition.notify_all()
        self._condition.release()

    def withdraw(self, amount):
        self._condition.acquire()
        while self._balance < amount:
            self._condition.wait()
        self._balance -= amount
        print(self._balance)
        self._condition.release()

#-----------------------------------------------------------------------

class DepositThread (threading.Thread):

    def __init__(self, bank_acct):
        threading.Thread.__init__(self)
        self._bank_acct = bank_acct

    def run(self):
        for _ in range(10):
            self._bank_acct.deposit(1)

#-----------------------------------------------------------------------

class WithdrawThread (threading.Thread):

    def __init__(self, bank_acct):
        threading.Thread.__init__(self)
        self._bank_acct = bank_acct

    def run(self):
        for _ in range(5):
            self._bank_acct.withdraw(2)

#-----------------------------------------------------------------------

def main():
    bank_acct = BankAcct()

    deposit_thread = DepositThread(bank_acct)
    withdraw_thread = WithdrawThread(bank_acct)

    deposit_thread.start()
    withdraw_thread.start()

    deposit_thread.join()
    withdraw_thread.join()

    print('Final balance:', bank_acct.get_balance())

#-----------------------------------------------------------------------

if __name__ == '__main__':
    main()
