// [wayne f16] fix halt instruction to give warning if it is 0xxx where x is not 0
// [wayne f16] todo: jump link and jump register

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;

import com.google.common.collect.ImmutableMap;

import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.util.List;
import java.util.Map;
import java.io.File;

public class ToyPseudocodeCheck extends AbstractFileSetCheck {


    // Keys pointing to the warning message text in "messages.properties"
    public static final String MSG_ARITHMETIC_OPERATOR     = "toy.pseudocode.operator";
    public static final String MSG_DESTINATION_REGISTER    = "toy.pseudocode.destination.register";
    public static final String MSG_SOURCE_REGISTER         = "toy.pseudocode.source.register";
    public static final String MSG_WRONG_FORMAT            = "toy.pseudocode.wrong.format";
    public static final String MSG_WRONG_FORMAT_LOW_MEMORY = "toy.pseudocode.wrong.format.low.memory";
    public static final String MSG_MEMORY_ADDRESS          = "toy.pseudocode.memory.address";
    public static final String MSG_LOAD_ADDRESS_CONSTANT   = "toy.pseudocode.load.address.constant";
    public static final String MSG_CONSTANT                = "toy.pseudocode.constant";
    public static final String MSG_CONSTANT_XTOY           = "toy.pseudocode.constant.xtoy";
    public static final String MSG_NOOP                    = "toy.pseudocode.noop";
    public static final String MSG_HALT                    = "toy.pseudocode.halt";
    public static final String MSG_EMPTY_COMMENT           = "toy.pseudocode.empty";

    // regular expression components
    private final String HEX                 = "[0-9A-Fa-f]";
    private final String MEMORY              = "((M|mem)\\[(" + HEX + HEX + ")\\])";
    private final String REGISTER            = "((R|reg)\\[(" + HEX + ")\\])";
    private final String MEMORY_INDIRECT     = "((M|mem)\\[(" + REGISTER + ")\\])";
    private final String WHITESPACE          = "\\s*";
    private final String GETS                = "(<\\-|=)";
    private final String ARITHMETIC_OPERATOR = "(\\^|\\+|\\-|\\&|<<|>>|>>>|\\*)";  // allow invalid operators here
    private final String COMPARISON_OPERATOR = "(>|==|=|>=|<=|<|!=)";              // allow invalid operators here
    private final String MINUS_SIGN          = "\\-";
    private final String ZERO                = "0";
    private final String STDIN               = "((?i)stdin|(?i)read|(?i)load)";
    private final String STDOUT              = "((?i)stdout|(?i)write|(?i)store|(?i)print)";
    private final String IF_KEYWORD          = "((?i)if)";
    private final String NOOP                = "((?i)noop|(?i)no\\-op)";
    private final String HALT_KEYWORD        = "((?i)halt)";
    private final String GOTO_KEYWORD        = "(pc\\s*=|PC\\s*=|(?i)goto|(?i)go to|pc\\s*<-|PC\\s*<-)";
    private final String CONSTANT_KEYWORD    = "((?i)constant|)";
    private final String START               = "^(";
    private final String END                 =  ").*$";

    private Map<Integer, String> operators = ImmutableMap.<Integer, String>builder()
        .put(1, "+")
        .put(2, "-")
        .put(3, "&")
        .put(4, "^")
        .put(5, "<<")
        .put(6, ">>")
        .put(12, "==")
        .put(13, ">")
        .build();

    // return a 16-bit integer corresponding to the 4-digit hex string s
    private static int fromHex(String s) {
        return Integer.parseInt(s, 16)  & 0xFFFF;
    }

    @Override
    protected void processFiltered(File file, FileText fileText) {
        for (int i = 0; i < fileText.size(); i++) {
            String line = fileText.get(i);
            int lineNo = i + 1;
            int columnNo = 0;
            ToyLine statement = new ToyLine(line, lineNo, columnNo);
            checkToyComment(statement);
        }
    }


    // check that TOY pseudocode matches the TOY instruction
    private void checkToyComment(ToyLine statement) {
        String regexp = "^" + WHITESPACE
                      + "(" + HEX + "{2}):" + WHITESPACE
                      + "(" + HEX + "{4})(.*$)";
        Pattern pattern = Pattern.compile(regexp);
        String line = statement.getText();
        Matcher matcher = pattern.matcher(line);
        if (matcher.find()) {
            int addr2 = fromHex(matcher.group(1));
            int inst = fromHex(matcher.group(2));
            String instruction = String.format("%02X: %04X", addr2, inst);
            String comment = matcher.group(3);
            int op    = (inst >> 12) &  15;    // get opcode (bits 12-15)
            int d     = (inst >>  8) &  15;    // get dest   (bits  8-11)
            int s     = (inst >>  4) &  15;    // get s      (bits  4- 7)
            int t     = (inst >>  0) &  15;    // get t      (bits  0- 3)
            int addr  = (inst >>  0) & 255;    // get addr   (bits  0- 7)

            // special case: only constants can be stored in 00 to 0F
            // must do before no-op special case
            if (addr2 >= 0x00 && addr2 < 0x0F) {
                // [wayne s17] temporarily removed until Visual X-TOY is updated
                /////////////// checkConstantComment(statement, comment, instruction, inst);
                return;
            }

            // special case for no-op comment
            if (isNoop(op, d, s, t, addr)) {
                checkNoopComment(statement, comment, instruction);
                return;
            }

            // special case for empty comment
            if (isEmptyComment(comment)) {
                reportEmptyComment(statement, instruction);
                return;
            }

            switch(op) {

                // halt instruction
                case  0: checkHaltComment(statement, comment, instruction, op, d, s, t);
                         break;

                // arithmetic-logic operations
                case  1:
                case  2:
                case  3:
                case  4:
                case  5:
                case  6: checkArithmeticLogicComment(statement, comment, instruction, op, d, s, t);
                         break;

                // load address
                case  7: checkLoadAddressComment(statement, comment, instruction, op, d, addr);
                         break;

                // load and store
                case  8:
                case  9: checkLoadStoreComment(statement, comment, instruction, op, d, addr);
                         break;

                // load indirect and store indirect
                case 10: 
                case 11: checkLoadStoreIndirectComment(statement, comment, instruction, op, d, t);
                         break;

                // branch zero and positive
                case 12:
                case 13: checkBranchComment(statement, comment, instruction, op, d, addr);
                         break;

                // TODO: jump register and jump and link
                case 14: break;
                case 15: break;
            }


        }
    }

    // check that the comment for 00 to 0F is the value of the constant
    private void checkConstantComment(ToyLine statement, String comment, String instruction, int inst) {
        comment = comment.trim();
        short instShort = (short) inst;

        String instStringHex       = String.format("%X", inst);
        String instStringDecimal   = String.format("%d", instShort);
        String instStringBinary    = Integer.toBinaryString(inst);
        String instStringBinary16  = String.format("%16s", Integer.toBinaryString(instShort)).replace(" ", "0");
        instStringBinary16 = instStringBinary16.substring(instStringBinary16.length() - 16);
        instStringBinary16 = instStringBinary16.substring(0,  4) + " "
                           + instStringBinary16.substring(4,  8) + " "
                           + instStringBinary16.substring(8, 12) + " "
                           + instStringBinary16.substring(12, 16);
        String instString4 = String.format("%04X", inst);


        // allow X-TOY comments of the form '(1111 1111 1111 1111,     -1)'
        String regexp1 = START
                       + "\\("
                       + "[01]{4} [01]{4} [01]{4} [01]{4},\\s+-?[0-9]*"
                       + "\\)"
                       + END;
        Pattern pattern1 = Pattern.compile(regexp1);
        Matcher matcher1 = pattern1.matcher(comment);
        if (matcher1.find()) {
            String shortComment = matcher1.group(1);
            String normalizedComment = shortComment.replaceAll("\\s+", " ");
            if (!normalizedComment.equals("(" + instStringBinary16 + ", " + instStringDecimal + ")")) {
                log(statement.getLineNo(),
                    statement.getColumnNo(),
                    MSG_CONSTANT_XTOY,
                    instruction,
                    comment,
                    "(" + instStringBinary16 + ", " + instStringDecimal + ")");
            }
            return;
        }

        // also allow comments of the form "constant 0xABCD" or "constant 17"
        String regexp2 = START
                       + CONSTANT_KEYWORD
                       + WHITESPACE
                       + "((0x)?[0-9A-F]+|-?[0-9]+|(0b)?[01]+)"
                       + WHITESPACE
                       + END;

        Pattern pattern2 = Pattern.compile(regexp2);
        Matcher matcher2 = pattern2.matcher(comment);
        if (matcher2.find()) {
            String constant = matcher2.group(3);
            if (!constant.matches("(0x)?0*" + instStringHex) &&
                !constant.matches("(0b)?0*" + instStringBinary) &&
                !constant.matches(instStringDecimal)) {
                log(statement.getLineNo(),
                    statement.getColumnNo(),
                    MSG_CONSTANT,
                    instruction,
                    comment,
                    constant,
                    instString4,
                    instStringDecimal);
            }
        }

        else {
            reportCommentIsInWrongFormatLowMemory(statement, instruction, comment);
        }
    }

    // check that a no-op comment is associated with a no-op statement
    private void checkNoopComment(ToyLine statement, String comment, String instruction) {
        comment = comment.trim();
        String regexp = START
                      + NOOP
                      + END;

        Pattern pattern = Pattern.compile(regexp);
        Matcher matcher = pattern.matcher(comment);
        if (!matcher.find()) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_NOOP,
                instruction,
                comment);
        }
    }

    // check that a halt comment is associated with a halt statement
    private void checkHaltComment(ToyLine statement, String comment, String instruction, int op, int d, int s, int t) {

        // ignore all halt instructions except 0000
        if (d != 0 || s != 0 || t != 0)
            return;

        comment = comment.trim();
        String regexp = START
                      + HALT_KEYWORD
                      + END;

        Pattern pattern = Pattern.compile(regexp);
        Matcher matcher = pattern.matcher(comment);
        if (!matcher.find()) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_HALT,
                instruction,
                comment);
        }
    }

    // check that arithmetic-logic comment is associated with an arithmetic-logic statement
    private void checkArithmeticLogicComment(ToyLine statement, String comment, String instruction, int op, int d, int s, int t) {
        comment = comment.trim();

        // R[d] <- 0
        String regexp0 = START
                       + REGISTER   + WHITESPACE
                       + GETS       + WHITESPACE
                       + "0|00|0000"
                       + END;

        // R[d] <- R[s]
        String regexp1 = START
                       + REGISTER   + WHITESPACE
                       + GETS       + WHITESPACE
                       + REGISTER
                       + END;

        // R[d] <- -R[t]
        String regexp2 = START
                       + REGISTER   + WHITESPACE
                       + GETS       + WHITESPACE
                       + MINUS_SIGN + WHITESPACE
                       + REGISTER
                       + END;

        // R[d] <- R[s] op R[t]
        String regexp3 = START
                       + REGISTER   + WHITESPACE
                       + GETS       + WHITESPACE
                       + REGISTER   + WHITESPACE
                       + ARITHMETIC_OPERATOR + WHITESPACE
                       + REGISTER
                       + END;

        Pattern pattern0 = Pattern.compile(regexp0);
        Matcher matcher0 = pattern0.matcher(comment);

        Pattern pattern1 = Pattern.compile(regexp1);
        Matcher matcher1 = pattern1.matcher(comment);

        Pattern pattern2 = Pattern.compile(regexp2);
        Matcher matcher2 = pattern2.matcher(comment);

        Pattern pattern3 = Pattern.compile(regexp3);
        Matcher matcher3 = pattern3.matcher(comment);

        // general case (do this before R[d] <- R[s]
        if (matcher3.find()) {
            String shortComment = matcher3.group(1);
            String dComment = matcher3.group(4);
            String sComment = matcher3.group(8);
            String opComment = matcher3.group(9);
            String tComment = matcher3.group(12);
            checkOperator(statement, instruction, shortComment, op, opComment);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister(statement, instruction, shortComment, s, sComment);
            checkSourceRegister(statement, instruction, shortComment, t, tComment);
        }

        // special case when instruction reduces to R[d] <- 0 because s = t = 0
        else if (s == 0 && t == 0 && matcher0.find()) {
            String shortComment = matcher0.group(1);
            String dComment = matcher0.group(4);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
        }

        // special case when instruction reduces to R[d] <- 0 because s = 0 (left/right shift)
        else if (s == 0 && (op == 5 || op == 6) && matcher0.find()) {
            String shortComment = matcher0.group(1);
            String dComment = matcher0.group(4);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
        }

        // special case when instruction reduces to R[d] <- 0 because either s = 0 or t = 0 (and)
        else if ((s == 0 || t == 0) && op == 3 && matcher0.find()) {
            String shortComment = matcher0.group(1);
            String dComment = matcher0.group(4);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
        }

        // special case when instruction reduces to R[d] <- R[s] because t = 0 (add, subtract, xor, shift left, shift right)
        else if (t == 0 && op != 3 && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String dComment = matcher1.group(4);
            String sComment = matcher1.group(8);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister(statement, instruction, shortComment, s, sComment);
        }

        // special case when instruction reduces to R[d] <- R[t] because s = 0 (add, xor)
        else if (s == 0 && (op == 1 || op == 4) && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String dComment = matcher1.group(4);
            String tComment = matcher1.group(8);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister(statement, instruction, shortComment, t, tComment);
        }

        // special case when instruction reduces to R[d] <- R[t] because s = t (and)
        else if (s == t && op == 3 && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String dComment = matcher1.group(4);
            String tComment = matcher1.group(8);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister(statement, instruction, shortComment, t, tComment);
        }

        // special case when instruction reduces to R[d] <- -R[t] because s = 0
        else if (s == 0 && op == 2 && matcher2.find()) {
            String shortComment = matcher2.group(1);
            String dComment = matcher2.group(4);
            String tComment = matcher2.group(8);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister(statement, instruction, shortComment, t, tComment);
        }

        else {
            reportCommentIsInWrongFormat(statement, instruction, comment);
        }
    }

    // check load address
    private void checkLoadAddressComment(ToyLine statement, String comment, String instruction, int op, int d, int addr) {
        comment = comment.trim();

        String regexp = START    + WHITESPACE
                      + REGISTER + WHITESPACE
                      + GETS     + WHITESPACE
                      +"(" + HEX + "{1,4}" + ")"
                      + END;


        Pattern pattern = Pattern.compile(regexp);
        Matcher matcher = pattern.matcher(comment);

        if (matcher.find()) {
            String shortComment = matcher.group(1);
            String dComment = matcher.group(4);
            String addrComment = matcher.group(6);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkLoadAddressConstant(statement, instruction, shortComment, addr, addrComment);
        }

        else {
            reportCommentIsInWrongFormat(statement, instruction, comment);
        }
    }

    // check load/store instructions
    // TODO: FF
    private void checkLoadStoreComment(ToyLine statement, String comment, String instruction, int op, int d, int addr) {
        comment = comment.trim();

        // load
        String regexp1 = START    + WHITESPACE
                       + REGISTER + WHITESPACE
                       + GETS     + WHITESPACE
                       + MEMORY
                       + END;

        // store
        String regexp2 = START    + WHITESPACE
                       + MEMORY   + WHITESPACE
                       + GETS     + WHITESPACE
                       + REGISTER
                       + END;

        // standard input and output
        String regexp3 = START
                       + ".*?"        // reluctant quantifier
                       + REGISTER
                       + END;

        Pattern pattern1 = Pattern.compile(regexp1);
        Matcher matcher1 = pattern1.matcher(comment);
        Pattern pattern2 = Pattern.compile(regexp2);
        Matcher matcher2 = pattern2.matcher(comment);
        Pattern pattern3 = Pattern.compile(regexp3);
        Matcher matcher3 = pattern3.matcher(comment);

        // load
        if (op == 8 && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String dComment = matcher1.group(4);
            String addrComment = matcher1.group(8);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkMemoryAddress(statement, instruction, shortComment, addr, addrComment);
        }

        // store
        else if (op == 9 && matcher2.find()) {
            String shortComment = matcher2.group(1);
            String addrComment = matcher2.group(4);
            String dComment = matcher2.group(8);
            checkMemoryAddress(statement, instruction, shortComment, addr, addrComment);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
        }


        // read from standard input
        else if (op == 8 && addr == 255 && matcher3.find()) {
            String shortComment = matcher3.group(1);
            String dComment = matcher3.group(4);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);

            // must have a keyword stdin/load/read
            if (!comment.matches(".*" + STDIN + ".*")) {
                reportCommentIsInWrongFormat(statement, instruction, comment);
            }
        }

        // write to standard output
        else if (op == 9 && addr == 255 && matcher3.find()) {
            String shortComment = matcher3.group(1);
            String dComment = matcher3.group(4);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);

            // must have a keyword stdout/write/store/print
            if (!comment.matches(".*" + STDOUT + ".*")) {
                reportCommentIsInWrongFormat(statement, instruction, comment);
            }
        }

        else {
            reportCommentIsInWrongFormat(statement, instruction, comment);
        }
    }

    // check load indirect and store indirect instructions
    private void checkLoadStoreIndirectComment(ToyLine statement, String comment, String instruction, int op, int d, int t) {
        comment = comment.trim();

        // load indirect
        String regexp1 = START    + WHITESPACE
                       + REGISTER + WHITESPACE
                       + GETS     + WHITESPACE
                       + MEMORY_INDIRECT
                       + END;

        // store indirect
        String regexp2 = START           + WHITESPACE
                       + MEMORY_INDIRECT + WHITESPACE
                       + GETS            + WHITESPACE
                       + REGISTER
                       + END;


        Pattern pattern1 = Pattern.compile(regexp1);
        Matcher matcher1 = pattern1.matcher(comment);
        Pattern pattern2 = Pattern.compile(regexp2);
        Matcher matcher2 = pattern2.matcher(comment);

        // load indirect
        if (op == 10 && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String dComment = matcher1.group(4);
            String tComment = matcher1.group(11);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister     (statement, instruction, shortComment, t, tComment);
        }

        // general case
        else if (op == 11 && matcher2.find()) {
            String shortComment = matcher2.group(1);
            String tComment = matcher2.group(7);
            String dComment = matcher2.group(11);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkSourceRegister     (statement, instruction, shortComment, t, tComment);
        }
        else {
            reportCommentIsInWrongFormat(statement, instruction, comment);
        }
    }

    // check branch if positive and branch if zero instructions
    private void checkBranchComment(ToyLine statement, String comment, String instruction, int op, int d, int addr) {
        comment = comment.trim();

        // special case for idiom C0xx
        String regexp1 = START + WHITESPACE
                       + GOTO_KEYWORD + WHITESPACE
                       + "(" + HEX + HEX + ")"
                       + END;

        // general case for Cxyz and Dxyz
        String regexp2 = START               + WHITESPACE
                       + IF_KEYWORD          + WHITESPACE
                       + "\\("               + WHITESPACE
                       + REGISTER            + WHITESPACE
                       + COMPARISON_OPERATOR + WHITESPACE
                       + ZERO                + WHITESPACE
                       + "\\)"               + WHITESPACE
                       + GOTO_KEYWORD        + WHITESPACE
                       + "(" + HEX + HEX + ")"
                       + END;


        Pattern pattern1 = Pattern.compile(regexp1);
        Matcher matcher1 = pattern1.matcher(comment);
        Pattern pattern2 = Pattern.compile(regexp2);
        Matcher matcher2 = pattern2.matcher(comment);

        // special case for idiom C0xx
        if (d == 0 && op == 12 && matcher1.find()) {
            String shortComment = matcher1.group(1);
            String addrComment = matcher1.group(3);
            checkMemoryAddress(statement, instruction, shortComment, addr, addrComment);
        }

        // general case
        else if (matcher2.find()) {
            String shortComment = matcher2.group(1);
            String dComment = matcher2.group(5);
            String opComment = matcher2.group(6);
            String addrComment = matcher2.group(8);
            checkOperator(statement, instruction, shortComment, op, opComment);
            checkDestinationRegister(statement, instruction, shortComment, d, dComment);
            checkMemoryAddress(statement, instruction, shortComment, addr, addrComment);
        }
        else {
            reportCommentIsInWrongFormat(statement, instruction, comment);
        }
    }

    // check if comment uses the correct arithmetic/logic/comparison operator
    private void checkOperator(ToyLine statement, String instruction, String comment, int op, String opComment) {
        if (!operators.get(op).equals(opComment)) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_ARITHMETIC_OPERATOR,
                instruction,
                comment,
                opComment,
                operators.get(op));
        }
    }

    // wrong format for memory locations 0x00 to 0x0F, which should store only constants
    private void reportCommentIsInWrongFormatLowMemory(ToyLine statement, String instruction, String comment) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_WRONG_FORMAT_LOW_MEMORY,
                instruction,
                comment);
    }

    // wrong format for memory locations 0x10 to 0xFE, which should store only instructions
    private void reportCommentIsInWrongFormat(ToyLine statement, String instruction, String comment) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_WRONG_FORMAT,
                instruction,
                comment);
    }

    private void checkDestinationRegister(ToyLine statement, String instruction, String comment, int d, String dComment) {
        String destinationRegister = String.format("%01X", d);
        if (!destinationRegister.equalsIgnoreCase(dComment)) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_DESTINATION_REGISTER,
                instruction,
                comment,
                dComment,
                destinationRegister);
        }
    }

    // [wayne s16] should we allow two source registers in opposite order for commutative operators (+, &, and ^) ???
    private void checkSourceRegister(ToyLine statement, String instruction, String comment, int s, String sComment) {
        String sourceRegister = String.format("%01X", s);
        if (!sourceRegister.equalsIgnoreCase(sComment)) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_SOURCE_REGISTER,
                instruction,
                comment,
                sComment,
                sourceRegister);
        }
    }

    private void checkMemoryAddress(ToyLine statement, String instruction, String comment, int addr, String addrComment) {
        String addrString = String.format("%02X", addr);
        if (!addrString.equalsIgnoreCase(addrComment)) {
            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_MEMORY_ADDRESS,
                instruction,
                comment,
                addrComment,
                addrString);
        }
    }

    private void checkLoadAddressConstant(ToyLine statement, String instruction, String comment, int addr, String addrComment) {
        String addrString1 = String.format("%01X", addr);
        String addrString2 = String.format("%02X", addr);
        String addrString3 = String.format("%03X", addr);
        String addrString4 = String.format("%04X", addr);

        if ((!addrString1.equalsIgnoreCase(addrComment)) &&
            (!addrString2.equalsIgnoreCase(addrComment)) &&
            (!addrString3.equalsIgnoreCase(addrComment)) &&
            (!addrString4.equalsIgnoreCase(addrComment))) {

            log(statement.getLineNo(),
                statement.getColumnNo(),
                MSG_LOAD_ADDRESS_CONSTANT,
                instruction,
                comment,
                addrComment,
                addrString4);
        }
    }

    private void reportEmptyComment(ToyLine statement, String instruction) {
        log(statement.getLineNo(),
            statement.getColumnNo(),
            MSG_EMPTY_COMMENT,
            instruction);
    }

    // is the instruction a no-op?
    private boolean isNoop(int op, int d, int s, int t, int addr) {
        if (op == 1 && d == s && t == 0) return true;   // R[x] <- R[x] + 0
        if (op == 1 && s == 0 && d == t) return true;   // R[x] <- 0 + R[x]
        if (op == 2 && s == d && t == 0) return true;   // R[x] <- R[x] - 0
        if (op == 3 && s == d && s == t) return true;   // R[x] <- R[x] & R[x]
        if (op == 5 && s == d && t == 0) return true;   // R[x] <- R[x] << 0
        if (op == 6 && s == d && t == 0) return true;   // R[x] <- R[x] >> 0
        if (op == 13 && d == 0) return true;            // if (R[0] > 0) pc <- xy
        if (d == 0 && ((op >= 1 && op <= 8) || (op == 10))) return true;  // R[0] <- ...
        return false;
    }

    // is the comment empty?
    private boolean isEmptyComment(String comment) {
        return comment.trim().length() == 0;
    }
}
