{
  open Lexing
let incr_linenum lexbuf =
    let pos = lexbuf.lex_curr_p in
    lexbuf.lex_curr_p <- { pos with
                           pos_lnum = pos.pos_lnum + 1;
                           pos_bol = pos.pos_cnum;
                         }

  let unexpected_char c lexbuf =
    Printf.printf "Char %s is unexpected at (line:%d, column:%d).\n" 
                  (Char.escaped c) 
                  (lexeme_start_p lexbuf).pos_lnum
                  ((lexeme_start_p lexbuf).pos_cnum - (lexeme_start_p lexbuf).pos_bol + 1)
type token =
  | IDENT of string
  | INT of int
  | EQ
  | SEMI
  | IFLZ
  | WHILELZ
  | LPAREN
  | RPAREN
  | LBRACE
  | RBRACE
  | ADD
  | MUL
  | NEG
  | RETURN

let show_token = function
  | IDENT x   -> "IDENT " ^ x
  | INT x     -> "INT " ^ (string_of_int x)
  | EQ        -> "EQ"
  | SEMI      -> "SEMI"
  | IFLZ      -> "IFLZ"
  | WHILELZ   -> "WHILELZ"
  | LPAREN    -> "LPAREN"
  | RPAREN    -> "RPAREN"
  | LBRACE    -> "LBRACE"
  | RBRACE    -> "RBRACE"
  | ADD	      -> "ADD"	    
  | MUL	      -> "MUL"	    
  | NEG	      -> "NEG"
  | RETURN    -> "RETURN"

exception End_of_file
}

let alpha = ['a'-'z''A'-'Z']
let digit = ['0'-'9']

rule lex = parse
  | "return" { RETURN }
  | "iflz" { IFLZ }
  | "whilelz" { WHILELZ }
  | ';' { SEMI }
  | '+' { ADD }
  | '*' { MUL }
  | '(' { LPAREN }
  | ')' { RPAREN }
  | '{' { LBRACE }
  | '}' { RBRACE }
  | '=' { EQ }
  | '-'?(digit|['1'-'9']digit+) as n { INT (int_of_string n) }
  | '-' { NEG }
  | alpha(alpha|digit|'_')* as id { IDENT id }
  | [' ''\t''\r']+ { lex lexbuf } (* ignore whitespace *)
  | '\n' { incr_linenum lexbuf; lex lexbuf } (* ignore newlines, but increment line num *)
  | eof { raise End_of_file }
  | _ as c 
      { unexpected_char c lexbuf; exit 1 }

{
  let tokenize buf : token list =
    let rec go (tokens : token list) : token list =
      try
        go ((lex buf)::tokens)
      with End_of_file -> List.rev tokens
    in
    go []

let _ =
  let chan = open_in Sys.argv.(1) in
  let tokens = tokenize (from_channel chan) in
  close_in chan;
  Format.printf "@[<hov 1>";
  List.iter (fun tok -> Format.printf "%s,@;" (show_token tok)) tokens;
  Format.printf "end of file@]@;"
;;

}
