/*
 * 29/01/00		Initial version. mdm@techie.com
/*-----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------
 */

/*
 * July 24th, 2003
 * This class was heavily modified by Bradley Zankel in order to allow
 * it to serve as a library for a Digital Signal Processing assignment
 * for a Computer Science course at Princeton University
 */

package javazoom.jl.player;

import java.io.*;
import javazoom.jl.decoder.*;
import java.net.URL;
import java.net.URLConnection;

/**
 * The <code>Player</code> class implements a simple player for playback
 * of an MPEG audio stream.
 *
 * @author	Mat McGowan
 * @since	0.0.8
 */

// REVIEW: the audio device should not be opened until the
// first MPEG audio frame has been decoded.
public class Player
{
	/**
	 * The current frame number.
	 */
	private int frame = 0;

	/**
	 * The MPEG audio bitstream.
	 */
	// javac blank final bug.
	/*final*/ private Bitstream		bitstream;

	/**
	 * The MPEG audio decoder.
	 */
	/*final*/ private Decoder		decoder;

	/**
	 * The AudioDevice the audio samples are written to.
	 */
	private AudioDevice	audio;

	/**
	 * Has the player been closed?
	 */
	private boolean		closed = false;

	/**
	 * Has the player played back all frames from the stream?
	 */
	private boolean		complete = false;

	private int			lastPosition = 0;

	private int			numchans = 0;

	// The following variables are additions by Bradley Zankel
	private short[]		buffer = null;

	private static Player player = null;

	private static boolean calledLeft = true, calledRight = true, empty = false;

	/**
	 * Creates a new <code>Player</code> instance.
	 */
	public Player(InputStream stream) throws JavaLayerException
	{
		this(stream, null);
	}

	public Player(InputStream stream, AudioDevice device) throws JavaLayerException
	{
		bitstream = new Bitstream(stream);
		decoder = new Decoder();

		if (device!=null)
		{
			audio = device;
		}
		else
		{
			FactoryRegistry r = FactoryRegistry.systemRegistry();
			audio = r.createAudioDevice();
		}
		audio.open(decoder);
	}

	public void play() throws JavaLayerException
	{
		play(Integer.MAX_VALUE);
	}

	/**
	 * Plays a number of MPEG audio frames.
	 *
	 * @param frames	The number of frames to play.
	 * @return	true if the last frame was played, or false if there are
	 *			more frames.
	 */
	public boolean play(int frames) throws JavaLayerException
	{
		boolean ret = true;

		while (frames-- > 0 && ret)
		{
			ret = decodeFrame();
		}

		if (!ret)
		{
			// last frame, ensure all data flushed to the audio device.
			AudioDevice out = audio;
			if (out!=null)
			{
				out.flush();
				synchronized (this)
				{
					complete = (!closed);
					end();
				}
			}
		}
		return ret;
	}

	/**
	 * Cloases this player. Any audio currently playing is stopped
	 * immediately.
	 */
	public synchronized void end()
	{
		AudioDevice out = audio;
		if (out!=null)
		{
			closed = true;
			audio = null;
			// this may fail, so ensure object state is set up before
			// calling this method.
			out.close();
			lastPosition = out.getPosition();
			try
			{
				bitstream.close();
			}
			catch (BitstreamException ex)
			{
			}
		}
	}

	/**
	 * Returns the completed status of this player.
	 *
	 * @return	true if all available MPEG audio frames have been
	 *			decoded, or false otherwise.
	 */
	public synchronized boolean isComplete()
	{
		return complete;
	}

	/**
	 * Retrieves the position in milliseconds of the current audio
	 * sample being played. This method delegates to the <code>
	 * AudioDevice</code> that is used by this player to sound
	 * the decoded audio samples.
	 */
	public int getPosition()
	{
		int position = lastPosition;

		AudioDevice out = audio;
		if (out!=null)
		{
			position = out.getPosition();
		}
		return position;
	}

	/**
	 * Decodes a single frame.
	 *
	 * @return true if there are no more frames to decode, false otherwise.
	 */
	protected boolean decodeFrame() throws JavaLayerException
	{
		try
		{
			AudioDevice out = audio;
			if (out==null)
				return false;

			Header h = bitstream.readFrame();

			if (h==null)
				return false;

			// sample buffer set when decoder constructed
			SampleBuffer output = (SampleBuffer)decoder.decodeFrame(h, bitstream);

			synchronized (this)
			{
				out = audio;
				if (out!=null)
				{
					out.write(output.getBuffer(), 0, output.getBufferLength());
				}
			}

			bitstream.closeFrame();
		}
		catch (RuntimeException ex)
		{
			throw new JavaLayerException("Exception decoding audio frame", ex);
		}
/*
		catch (IOException ex)
		{
			System.out.println("exception decoding audio frame: "+ex);
			return false;
		}
		catch (BitstreamException bitex)
		{
			System.out.println("exception decoding audio frame: "+bitex);
			return false;
		}
		catch (DecoderException decex)
		{
			System.out.println("exception decoding audio frame: "+decex);
			return false;
		}
*/
		return true;
	}

	/*** ADD-ONS BY BRADLEY ZANKEL TO ALLOW THE USER TO RECEIVE THE RAW DATA ***/
	// extra constructors
	public Player(String filename) {
		try {
			FileInputStream fin = new FileInputStream(filename);
			BufferedInputStream bin = new BufferedInputStream(fin);

			bitstream = new Bitstream(bin);
			decoder = new Decoder();

			FactoryRegistry r = FactoryRegistry.systemRegistry();
			audio = r.createAudioDevice();

			audio.open(decoder);
		} catch (JavaLayerException e) {
			System.err.println("Error opening audio device!");
			System.exit(-1);
		} catch (IOException e) {
			System.err.println("Error processing file: "+filename);
			System.exit(-1);
		}
	}

	public Player() {
		try  {
			URL url = getClass().getResource("dummy.mp3");
			URLConnection site = url.openConnection();
			InputStream is = site.getInputStream();
			BufferedInputStream bin = new BufferedInputStream(is);

			bitstream = new Bitstream(bin);
			decoder = new Decoder();

			FactoryRegistry r = FactoryRegistry.systemRegistry();
			audio = r.createAudioDevice();

			audio.open(decoder);
			init();
		} catch (JavaLayerException e) {
			System.err.println("Error opening audio device!");
			System.exit(-1);
		} catch (IOException e) {
			System.err.println("Error accessing default mp3 header.");
			System.exit(-1);
		}
	}

	// Static methods
	public static boolean open(String filename) {
		player = new Player(filename);

		return (player != null);
	}

	public static boolean open() {
		player = new Player();

		return (player != null);
	}

	public static void initialize() {
		player.init();
	}

	public static boolean playWave(short[] left, short[] right) {
		return player.playW(left, right);
	}

	public static boolean playWave(short[] buffer) {
		return player.playW(buffer);
	}

	public static boolean playWave(short[] buffer, int offset, int length) {
		return player.playW(buffer, offset, length);
	}

	public static void close() {
		player.finish();
	}

	// same as decodeFrame except this returns the audio raw data instead of
	// true and null instead of false.
	public short[] playFrame() {
			short[] buffer = getFrame();

			if (buffer == null)
				return null;

			playWave(buffer, 0, buffer.length);

			return buffer;
	}

	// simply get the data from the frame, and return the decoded short[]
	public short[] getFrame() {
		try {
			AudioDevice out = audio;
			if (out == null)
				return null;

			Header h = bitstream.readFrame();

			if (h == null)
				return null;

			SampleBuffer output = (SampleBuffer)decoder.decodeFrame(h, bitstream);
			numchans = output.getChannelCount();

			bitstream.closeFrame();
			return output.getBuffer();
		} catch (Exception ex) {
			System.err.println("Exception decoding audio frame");
			ex.printStackTrace();
			return null;
		}
	}

    public static boolean isEmpty() {
		return empty;
	}

	public static short[] getLeftChannel() {
		if (calledLeft) {
			calledRight = false;
			if ((player.buffer = player.getFrame()) == null) {
				empty = true;
				player.buffer = new short[2304];
			}
		} else
			calledLeft = true;

		short[] left = new short[player.buffer.length/2];

		for (int i = 0; i < left.length; i++)
			left[i] = player.buffer[2*i];

		return left;
	}

	public static short[] getRightChannel() {
		if (calledRight) {
			calledLeft = false;
			if ((player.buffer = player.getFrame()) == null) {
				empty = true;
				player.buffer = new short[2304];
			}
		} else
			calledRight = true;

		short[] right = new short[player.buffer.length/2];

		for (int i = 0; i < right.length; i++)
			right[i] = player.buffer[2*i+1];

		return right;
	}

	public static short[] nextFrame() {
		return player.getFrame();
	}

	public boolean init() {
		try {
			AudioDevice out = audio;
			if (out == null)
				return false;

			Header h = bitstream.readFrame();

			if (h == null)
				return false;

			decoder.decodeFrame(h, bitstream);

			bitstream.unreadFrame();
			return true;
		} catch (Exception ex) {
			System.err.println("Exception intializing AudioDevice");
			ex.printStackTrace();
			return false;
		}
	}

	public boolean playW(short[] left, short[] right) {
		if (right.length != left.length) {
			System.err.println("Error, left stereo channel length and right "
								+"stereo channel length don't match");
			return false;
		}

		int samples = left.length;
		short[] buffer = new short[2*samples];

		for (int i = 0; i < samples; i++) {
			buffer[2*i] = left[i];
			buffer[2*i+1] = right[i];
		}

		return playW(buffer, 0, buffer.length);
	}

	public boolean playW(short[] buffer) {
		return playW(buffer, 0, buffer.length);
	}

	public boolean playW(short[] buffer, int offset, int length) {
		try {
			AudioDevice out = audio;
			if (out == null)
				return false;

			synchronized (this) {
				out = audio;
				if (out != null)
					out.write(buffer, offset, length);
			}
		} catch (Exception e) {
			System.err.println("Exception playing sound wave");
			e.printStackTrace();
			return false;
		}

		return true;
	}

	public int getChanCount() {
		return numchans;
	}

	// Cleanup after we're done playing the audio file
	public void finish() {
		try {
			AudioDevice out = audio;

			if (out != null) {
				out.flush();
				synchronized (this) {
					complete = (!closed);
					end();
				}
			}
		} catch (Exception ex) {
			System.out.println("Error closing audio device");
			System.exit(-1);
		}
	}
}
