// *** REFERENCE SOLUTION
public class TigerWallet {

    private static final int COST_PER_TIGERCOIN = 100;  // dollars per tigercoin
    private static final int TRANSACTION_FEE = 3;       // cost per transaction

    private final String network;   // network
    private int cash;               // dollars
    private int coin;               // tigercoins

    // creates a new wallet in given network, with given cash and coin
    public TigerWallet(String network, int cash, int coin) {
        this.network = network;
        this.cash = cash;
        this.coin = coin;
    }

    // returns the amount of cash in this wallet
    public int getCash() {
        return cash;
    }

    // returns the number of tigercoins in this wallet
    public int getCoin() {
        return coin;
    }

    // deposits the specified amount of cash in this wallet
    public void depositCash(int amount) {
        cash += amount;
    }

    // buys the specified number of tigercoins
    public void buyCoin(int amount) {
        int cost = (COST_PER_TIGERCOIN * amount) + TRANSACTION_FEE;

        if (cost > cash) {
            throw new IllegalArgumentException("insufficient cash");
        }

        cash -= cost;
        coin += amount;
    }

    // transfer the specified number of tigercoins from this wallet to the other one
    public void transferCoinTo(int amount, TigerWallet to) {
        if (amount > coin) {
            throw new IllegalArgumentException("insufficient coin");
        }

        // out-of-network transfer fee
        if (!network.equals(to.network)) {
            if (cash < TRANSACTION_FEE) {
                throw new IllegalArgumentException("insufficient cash");
            }
            else cash -= TRANSACTION_FEE;
        }

        this.coin -= amount;
        to.coin += amount;
    }

    // merge the wallets into this one, assuming they are all on same network
    public void mergeWith(TigerWallet[] wallets) {

        // check that all accounts are on same network
        for (int i = 0; i < wallets.length; i++) {
            if (!network.equals(wallets[i].network))
                throw new IllegalArgumentException("wallets must be in same network");
        }

        // check that no aliased wallets
        for (int i = 0; i < wallets.length; i++) {
            if (this == wallets[i]) {
                throw new IllegalArgumentException("alias wallets");
            }
            for (int j = i+1; j < wallets.length; j++) {
                if (wallets[i] == wallets[j]) {
                    throw new IllegalArgumentException("alias wallets");
                }
            }
        }

        // merge the accounts
        for (int i = 0; i < wallets.length; i++) {
            this.cash += wallets[i].cash;
            this.coin += wallets[i].coin;
            wallets[i].cash = 0;
            wallets[i].coin = 0;
        }
    }

    // returns a string representation of this wallet
    public String toString() {
        return "(" + network + ", " + cash + ", " + coin + ")";
    }

    // unit tests the TigerWallet data type
    public static void main(String[] args) {
        TigerWallet aja = new TigerWallet("Mathey", 100, 10);
        TigerWallet bob = new TigerWallet("Forbes", 300,  1);
        TigerWallet cai = new TigerWallet("Mathey",   0,  6);
        TigerWallet dee = new TigerWallet("Mathey", 200,  0);

        // print Aja’s cash and coin
        StdOut.println("Aja cash: " + aja.getCash());    // Aja cash: 100
        StdOut.println("Aja coin: " + aja.getCoin());    // Aja coin: 10

        // perform some transactions
        aja.depositCash(1000);
        aja.buyCoin(5);
        aja.transferCoinTo(2, cai);

        // print wallets
        StdOut.println("Aja: " + aja);    // Aja: (Mathey, 597, 13)
        StdOut.println("Bob: " + bob);    // Bob: (Forbes, 300, 1)
        StdOut.println("Cai: " + cai);    // Cai: (Mathey, 0, 8)
        StdOut.println("Dee: " + dee);    // Dee: (Mathey, 200, 0)
        StdOut.println();

        // merge Mathey wallets into Aja's wallet
        TigerWallet[] wallets = { cai, dee };
        aja.mergeWith(wallets);

        // print wallets
        StdOut.println("Aja: " + aja);    // Aja: (Mathey, 797, 21)
        StdOut.println("Bob: " + bob);    // Bob: (Forbes, 300, 1)
        StdOut.println("Cai: " + cai);    // Cai: (Mathey, 0, 0)
        StdOut.println("Dee: " + dee);    // Dee: (Mathey, 0, 0)
        StdOut.println();
    }
}
