package edu.princeton.cs.lift.checkstyle;

import java.util.TreeMap;

import com.puppycrawl.tools.checkstyle.api.AbstractCheck;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;
import com.puppycrawl.tools.checkstyle.utils.CheckUtil;
import com.puppycrawl.tools.checkstyle.utils.CommonUtil;

/**
 * <p>
 * Checks that specified numeric literals appear at most the
 * specified number of times.
 * </p>
 * <p>Check have following options:
 * ignoreHashCodeMethod - ignore numeric literals in hashCode methods;
 * ignoreMainMethod - ignore numeric literals in main methods;
 * <p>
 */
public class NumericLiteralCountCheck extends AbstractCheck {

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY = "numeric.literal.count";

    /** key = numeric literal; value = frequency of occurrence. */
    private TreeMap<Double, Integer> frequency;

    /** The numeric literals. */
    private double[] numericLiterals = { };

    /** Whether to ignore magic numbers in a hash code method. */
    private boolean ignoreHashCodeMethod;

    /** Whether to ignore magic numbers in a main method. */
    private boolean ignoreMainMethod;

    // maximum number of occurrences of numeric literal
    private int max = Integer.MAX_VALUE;

    public void setMax(int value) {
        max = value;
    }

    @Override
    public int[] getDefaultTokens() {
        return getAcceptableTokens();
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {
            TokenTypes.NUM_DOUBLE,
            TokenTypes.NUM_FLOAT,
            TokenTypes.NUM_INT,
            TokenTypes.NUM_LONG,
        };
    }

    @Override
    public int[] getRequiredTokens() {
        return CommonUtil.EMPTY_INT_ARRAY;
    }

    @Override
    public void visitToken(DetailAST ast) {
        if (ignoreMainMethod && Utilities.isInMainMethod(ast)) return;
        else if (ignoreHashCodeMethod && Utilities.isInHashCodeMethod(ast)) return;
        double number = extractNumber(ast);
        if (!frequency.containsKey(number)) return;
        frequency.put(number, frequency.get(number) + 1);
        /* print error message for each excess
        if (frequency.get(number) > max)
            reportMagicNumber(ast);
        */
    }

    /**
     * Reports ast as a numeric literal, includes unary operators as needed.
     * @param ast the AST node that contains the number to report
     */
    private void reportMagicNumber(DetailAST ast) {
        String text = ast.getText();
        final DetailAST parent = ast.getParent();
        DetailAST reportAST = ast;
        if (parent.getType() == TokenTypes.UNARY_MINUS) {
            reportAST = parent;
            text = "-" + text;
        }
        else if (parent.getType() == TokenTypes.UNARY_PLUS) {
            reportAST = parent;
            text = "+" + text;
        }

        log(reportAST.getLineNo(),
            reportAST.getColumnNo(),
            MSG_KEY,
            text,
            max);
        }

    /**
     * Extract the numeric literal from the AST.
     * @param ast the AST to check
     * @return the number
     */
    private double extractNumber(DetailAST ast) {
        double value = CheckUtil.parseDouble(ast.getText(), ast.getType());
        final DetailAST parent = ast.getParent();
        if (parent.getType() == TokenTypes.UNARY_MINUS) {
            value = -1 * value;
        }
        return value;
    }

    /**
     * Sets the numbers to ignore in the check.
     * BeanUtil converts numeric token list to double array automatically.
     * @param list list of numbers to ignore.
     */
    public void setNumericLiterals(double... list) {
        numericLiterals = list.clone();
    }

    @Override
    public void beginTree(DetailAST rootAST) {
        frequency = new TreeMap<Double, Integer>();
        for (double number : numericLiterals)
            frequency.put(number, 0);
    }

    private static boolean isMathematicalInteger(double x) {
        return Double.isFinite(x) && x == Math.rint(x);
    }

    @Override
    public void finishTree(DetailAST rootAST) {
        for (double number : numericLiterals) {
            if (frequency.get(number) > max) {

                // messages.properties seems to truncate doubles to integers
                // hack to print 3.0 as 3
                String numberAsString = number + "";
                if (isMathematicalInteger(number))
                    numberAsString = "" + (long) number;

                log(rootAST.getLineNo(),
                    rootAST.getColumnNo(),
                    MSG_KEY,
                    numberAsString,
                    frequency.get(number),
                    max);
            }
        }
    }


    /**
     * Set whether to ignore hashCode methods.
     * @param ignore decide whether to ignore hash code methods
     */
    public void setIgnoreHashCodeMethod(boolean ignore) {
        ignoreHashCodeMethod = ignore;
    }


    /**
     * Set whether to ignore main methods.
     * @param ignore decide whether to ignore hash code methods
     */
    public void setIgnoreMainMethod(boolean ignore) {
        ignoreMainMethod = ignore;
    }
}
