/*************************************************************************
 *  Basically the same as MagicNumberCheck.java, except that it allows
 *  you to ignore magic numbers in main().
 *
 * see https://github.com/checkstyle/checkstyle/blob/master/src/main/java/com/puppycrawl/tools/checkstyle/checks/coding/MagicNumberCheck.java
 *
 *************************************************************************/

////////////////////////////////////////////////////////////////////////////////
// checkstyle: Checks Java source code for adherence to a set of rules.
// Copyright (C) 2001-2016 the original author or authors.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
////////////////////////////////////////////////////////////////////////////////

package edu.princeton.cs.lift.checkstyle;

import java.util.Arrays;

import com.puppycrawl.tools.checkstyle.api.AbstractCheck;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;
import com.puppycrawl.tools.checkstyle.utils.CheckUtil;
import com.puppycrawl.tools.checkstyle.utils.CommonUtil;

/**
 * <p>
 * Checks that only specified numeric literals appear in code.
 * By default, -1, 0, 1, and 2 are allowed.
 * </p>
 * <p>Check have following options:
 * ignoreHashCodeMethod - ignore magic numbers in hashCode methods;
 * ignoreMainMethod - ignore magic numbers in main methods;
 * <p>
 */
public class NumericLiteralCheck extends AbstractCheck {

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY = "numeric.literal";

    /** The numbers to ignore in the check, sorted. */
    private double[] ignoreNumbers = {-1, 0, 1, 2};

    /** Whether to ignore magic numbers in a hash code method. */
    private boolean ignoreHashCodeMethod;

    /** Whether to ignore magic numbers in a main method. */
    private boolean ignoreMainMethod;


    @Override
    public int[] getDefaultTokens() {
        return getAcceptableTokens();
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {
            TokenTypes.NUM_DOUBLE,
            TokenTypes.NUM_FLOAT,
            TokenTypes.NUM_INT,
            TokenTypes.NUM_LONG,
        };
    }

    @Override
    public int[] getRequiredTokens() {
        return CommonUtil.EMPTY_INT_ARRAY;
    }

    @Override
    public void visitToken(DetailAST ast) {
        if (ignoreMainMethod && Utilities.isInMainMethod(ast)) return;
        else if (ignoreHashCodeMethod && Utilities.isInHashCodeMethod(ast)) return;
        else if (isInIgnoreList(ast)) return;
        else reportMagicNumber(ast);
    }

    /**
     * Reports ast as a magic number, includes unary operators as needed.
     * @param ast the AST node that contains the number to report
     */
    private void reportMagicNumber(DetailAST ast) {
        String text = ast.getText();
        final DetailAST parent = ast.getParent();
        DetailAST reportAST = ast;
        if (parent.getType() == TokenTypes.UNARY_MINUS) {
            reportAST = parent;
            text = "-" + text;
        }
        else if (parent.getType() == TokenTypes.UNARY_PLUS) {
            reportAST = parent;
            text = "+" + text;
        }
        log(reportAST.getLineNo(),
                reportAST.getColumnNo(),
                MSG_KEY,
                text);
    }

    /**
     * Decides whether the number of an AST is in the ignore list of this
     * check.
     * @param ast the AST to check
     * @return true if the number of ast is in the ignore list of this check.
     */
    // see https://github.com/checkstyle/checkstyle/blob/master/src/main/java/com/puppycrawl/tools/checkstyle/checks/coding/MagicNumberCheck.java
    private boolean isInIgnoreList(DetailAST ast) {
        double value = CheckUtil.parseDouble(ast.getText(), ast.getType());
        final DetailAST parent = ast.getParent();
        if (parent.getType() == TokenTypes.UNARY_MINUS) {
            value = -1 * value;
        }
        return Arrays.binarySearch(ignoreNumbers, value) >= 0;
    }

    /**
     * Sets the numbers to ignore in the check.
     * BeanUtil converts numeric token list to double array automatically.
     * @param list list of numbers to ignore.
     */
    // see https://github.com/checkstyle/checkstyle/blob/master/src/main/java/com/puppycrawl/tools/checkstyle/checks/coding/MagicNumberCheck.java
    public void setIgnoreNumbers(double... list) {
        if (list.length == 0) {
            ignoreNumbers = CommonUtil.EMPTY_DOUBLE_ARRAY;
        }
        else {
            ignoreNumbers = new double[list.length];
            System.arraycopy(list, 0, ignoreNumbers, 0, list.length);
            Arrays.sort(ignoreNumbers);
        }
    }

    /**
     * Set whether to ignore hashCode methods.
     * @param ignore decide whether to ignore hash code methods
     */
    public void setIgnoreHashCodeMethod(boolean ignore) {
        ignoreHashCodeMethod = ignore;
    }


    /**
     * Set whether to ignore main methods.
     * @param ignore decide whether to ignore hash code methods
     */
    public void setIgnoreMainMethod(boolean ignore) {
        ignoreMainMethod = ignore;
    }
}
