public class LapTimer {

    // number of laps
    private int n;

    // cumulative time of the n laps
    private double cumulativeTime;

    // laps[i] = time for lap i (i between 0 and n-1)
    private double[] laps;

    // create a new lap timer that supports up to max laps
    public LapTimer(int max) {
        laps = new double[max];
    }

    // add a new lap with the specified time
    public void addLap(double time) {

        // invalid time
        if (time <= 0.0) {
            throw new IllegalArgumentException("invalid time: " + time);
        }

        // too many laps added
        if (n == laps.length) {
            throw new IllegalArgumentException("overflow error");
        }

        // add the lap
        cumulativeTime += time;
        laps[n++] = time;
    }

    // return the number of laps added
    public int count() {
        return n;
    }

    // return the cumulative time of all laps
    public double cumulativeTime() {
        return cumulativeTime;
    }

    // returns the fastest lap time; infinity if no laps
    public double fastestLap() {
        double fastest = Double.POSITIVE_INFINITY;
        for (int i = 0; i < n; i++) {
            if (laps[i] < fastest) {
                fastest = laps[i];
            }
        }
        return fastest;
    }

    // return the fastest cumulative of any k consecutive laps
    public double fastestMultiLap(int k) {
        double sum = 0.0;
        for (int i = 0; i < k; i++) {
            sum += laps[i];
        }
        double fastest = sum;

        // sum = sum of times of k consecutive laps, ending at index i
        for (int i = k; i < n; i++) {
            sum = sum - laps[i - k] + laps[i];
            if (sum < fastest) {
                fastest = sum;
            }
        }
        return fastest;
    }

    // return the longest strictly decreasing sequence
    public int longestDecreasingStreak() {
        if (n == 0) return 0;
        int longest = 1;
        int streak = 1;
        for (int i = 1; i < n; i++) {
            if (laps[i] < laps[i-1]) streak++;
            else streak = 1;
            longest = Math.max(streak, longest);
        }
        return longest;
    }

    // returns a string representation of this LapTimer
    public String toString() {
        String result =  "[";
        for (int i = 0; i < n-1; i++) {
            result += laps[i] + ", ";
        }
        if (n > 0) result += laps[n-1];
        result += "]";
        return result;
    }


    // unit test
    public static void main(String[] args) {

        // create a lap timer for Usain Bolt’s 100-meter world record,
        // dividing the race into 10 segments (or "laps")
        LapTimer timer = new LapTimer(10);
        timer.addLap(1.85);
        timer.addLap(1.02);
        timer.addLap(0.91);
        timer.addLap(0.87);
        timer.addLap(0.85);
        timer.addLap(0.82);
        timer.addLap(0.82);
        timer.addLap(0.82);
        timer.addLap(0.83);
        timer.addLap(0.90);

        // print statistics
        StdOut.println("number of laps  = " + timer.count());                   // 10
        StdOut.println("cumulative time = " + timer.cumulativeTime());          // 9.69
        StdOut.println("fastest lap     = " + timer.fastestLap());              // 0.82
        StdOut.println("fastest 50m     = " + timer.fastestMultiLap(5));        // 4.14
        StdOut.println("longest streak  = " + timer.longestDecreasingStreak()); // 6

        // print times: [1.85, 1.02, 0.91, 0.87, 0.85, 0.82, 0.82, 0.82, 0.83, 0.9]
        StdOut.println(timer);
    }
}
