/* ****************************************
 * Pedometer Reference Implementation
 * ****************************************/

public class Pedometer {
    private static final int M_KM = 1000; // 1000m / km
    private int cx; // keep track of current x position
    private int cy; //  keep track of current y position
    private int distance; // keep track of total distance traveled

    // creates a Pedometer at coordinate (cx, cy) where each unit represents 1 meter
    public Pedometer(int x, int y) {
        // explicitly initialize instance variables
        cx = x;
        cy = y;
        distance = 0;
    }

    // returns a String representation of this Pedometer that includes the 
    // current position and the distance traveled in kilometers and meters
    public Pedometer(int x, int y, Queue<String> moves) {
        // explicitly initialize instance variables
        cx = x;
        cy = y;
        distance = 0;

        // look at each move in the Queue
        for (String move : moves) {
            // must be only 2 characters!
            if (move.length() != 2)
                throw new IllegalArgumentException("Illegal moves format");

            // check if first char is a direction char
            int index = 0;
            char direction = move.charAt(index);
            if ((direction != 'E') && (direction != 'W') && (direction != 'S') 
                    && (direction != 'N'))
                throw new IllegalArgumentException("Illegal value for direction: "
                                                   + direction);

            // check if second char is a meter value between 0 & p
            index = 1; 
            char meters = move.charAt(index);
            if ((meters < '0') || (meters > '9'))
                throw new IllegalArgumentException("Illegal value for meters: "
                                                    + meters);
            // valid direction and meters value - so call move
            move(direction, Integer.parseInt("" + meters));
        }
    }

    // moves the Pedometer to a new position (nx, ny)
    public void moveTo(int nx, int ny) {
        // use Manhattan distance and update total distance
        distance += Math.abs(cx - nx) + Math.abs(cy - ny);
        // update current position
        cx = nx;
        cy = ny;

    }

    // returns the total distance in meters traveled by this Pedometer
    public int distanceTraveled() {
        return distance;
    }

    // returns a String representation of this Pedometer that includes the 
    // current position and the distance traveled in kilometers and meters
    public String toString() {
        String s = "";
        // calculate km and m
        int km = distance / M_KM;
        int m = distance % M_KM;
        // format the String
        s += "current:" + "(" + cx + "," + cy + ")/";
        s += "distance:" + km + "km," + m + "m";
        return s;
    }

    // moves the Pedometer: east -'E', west -'W', north -'N', or south -'S' by 
    // the given number of meters m (m >= 0) and updates the new position
    public void move(char direction, int meters) {
        // validate arguments
        if ((direction != 'E') && (direction != 'W') &&
            (direction != 'S') && (direction != 'N'))
            throw new IllegalArgumentException("Illegal value for direction: "
                                               + direction);
        if (meters < 0)
            throw new IllegalArgumentException("Illegal value for meters: " + meters);

        int oldX = cx; // save old values
        int oldY = cy;

        // move m meters in given direction
        if (direction == 'E')
            cx += meters;
        else if (direction == 'W')
            cx -= meters;
        else if (direction == 'N')
            cy += meters;
        else
            cy -= meters;

        // increment and add distance
        // alternate: distance += Math.abs(oldX − cx) + Math.abs(oldY − cy);
        distance += Math.abs(oldX - cx) + Math.abs(oldY - cy);
    }

    // compare if this Pedometer's distance is less than that one
    public boolean lessThan(Pedometer that) {
        return this.distance < that.distance;
    }

    // main method - directly tests all instance methods in this class
    public static void main(String[] args) {
        // Part I:
        Pedometer p1 = new Pedometer(0, 0);     // create a Pedometer object
        StdOut.println(p1);                     // test toString
        p1.moveTo(6, 7);                        // test moveTo
        StdOut.println(p1);                     // test toString
        StdOut.println(p1.distanceTraveled());  // test distanceTraveled
        Pedometer p2 = new Pedometer(0, 0);     // create a Pedometer object
        StdOut.println(p2);                     // test toString
        p2.moveTo(-1000, 2123);                 // test moveTo
        StdOut.println(p2);                     // test toString
        StdOut.println(p2.distanceTraveled());  // test distanceTraveled
        StdOut.println(p1.lessThan(p2));        // test lessThan - true
        StdOut.println(p2.lessThan(p1));        // test lessThan - false
        StdOut.println(p1.lessThan(p1));        // test lessThan - false
        Pedometer p3 = new Pedometer(0, 0);     // create a Pedometer object
        p3.move('E', 1);                        // test directional move
        StdOut.println(p3);                     // test toString
        p3.move('N', 1);                        // test directional move
        StdOut.println(p3);                     // test toString
        StdOut.println(p3.distanceTraveled());  // test distanceTraveled
        
        // Part II:
        Queue<String> moves = new Queue<String>();
        moves.enqueue("E1");
        moves.enqueue("S1");
        moves.enqueue("E3");
        moves.enqueue("W2");
        Pedometer p4 = new Pedometer(0, 0, moves);
        StdOut.println(p4);
    }

}
