/********************************************************************************
 *  Flags all instance variables, static variables, instance methods,
 *  static methods, enum, interface, nested classes, and (optionally) constants.
 ********************************************************************************/
package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import com.puppycrawl.tools.checkstyle.utils.CommonUtil;

public class NothingButMainCheck extends AbstractCheck {

    // If true, applies the check to static final variables.
    private boolean ignoreConstants = false;

    // If true, applies the check to static methods.
    private boolean ignoreStaticMethods = false;

    // If true, applies the check to static variables (not final).
    private boolean ignoreStaticVariables = false;

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_INSTANCE_VARIABLE = "nothing.but.main.instance.variable";
    public static final String MSG_INSTANCE_METHOD = "nothing.but.main.instance.method";
    public static final String MSG_STATIC_VARIABLE = "nothing.but.main.static.variable";
    public static final String MSG_STATIC_FINAL_VARIABLE = "nothing.but.main.static.final.variable";
    public static final String MSG_STATIC_METHOD = "nothing.but.main.static.method";
    public static final String MSG_ENUM = "nothing.but.main.enum";
    public static final String MSG_INTERFACE = "nothing.but.main.interface";
    public static final String MSG_NESTED_CLASS = "nothing.but.main.nested.class";

   /**
     * Sets whether we should apply the check to static final variables.
     *
     * @param ignore new value of the property.
     */
    public void setIgnoreConstants(boolean ignore) {
        ignoreConstants = ignore;
    }

   /**
     * Sets whether we should apply the check to static methods.
     *
     * @param ignore new value of the property.
     */
    public void setIgnoreStaticMethods(boolean ignore) {
        ignoreStaticMethods = ignore;
    }

   /**
     * Sets whether we should apply the check to static variables (not final).
     *
     * @param ignore new value of the property.
     */
    public void setIgnoreStaticVariables(boolean ignore) {
        ignoreStaticVariables = ignore;
    }

    @Override
    public int[] getDefaultTokens() {
        return new int[] {
            TokenTypes.METHOD_DEF,
            TokenTypes.VARIABLE_DEF,
            TokenTypes.ENUM_DEF,
            TokenTypes.CLASS_DEF,
            TokenTypes.INTERFACE_DEF,
        };
    }

    @Override
    public int[] getRequiredTokens() {
        return CommonUtil.EMPTY_INT_ARRAY;
    }

    @Override
    public int[] getAcceptableTokens() {
        return getDefaultTokens();
    }

    @Override
    public void visitToken(DetailAST ast) {
        String kind = null;

        // flag all methods except main()
        if (ast.getType() == TokenTypes.METHOD_DEF) {
            if (Utilities.isInstanceMethod(ast)) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_INSTANCE_METHOD);
            }
            else if (Utilities.isStaticMethod(ast) && !Utilities.isMainMethod(ast)) {
                if (!ignoreStaticMethods) {
                    log(ast.getLineNo(),
                        ast.getColumnNo(),
                        MSG_STATIC_METHOD);
                }
            }
        }

        // flag all instance and static variable definitions
        else if (ast.getType() == TokenTypes.VARIABLE_DEF) {
            if (Utilities.isInstanceVariable(ast)) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_INSTANCE_VARIABLE);
            }
            else if (Utilities.isStaticFinalVariable(ast)) {
                if (!ignoreConstants) {
                    log(ast.getLineNo(),
                        ast.getColumnNo(),
                        MSG_STATIC_FINAL_VARIABLE);
                }
            }

            else if (Utilities.isStaticVariable(ast)) {
                if (!ignoreStaticVariables) {
                    log(ast.getLineNo(),
                        ast.getColumnNo(),
                        MSG_STATIC_VARIABLE);
                }
            }
        }

        // flag all enum definitions
        else if (ast.getType() == TokenTypes.ENUM_DEF) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_ENUM);
        }

        // flag all interface definitions
        else if (ast.getType() == TokenTypes.INTERFACE_DEF) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_INTERFACE);
        }

        // flag all nested class definitions
        else if (ast.getType() == TokenTypes.CLASS_DEF) {
            if (Utilities.isNestedClass(ast)) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_NESTED_CLASS);
            }
        }
    }


}
