// REFERENCE IMPLEMENTATION
// A. Kaplan

public class DNA {
    private final String name;  // DNA name
    private final String seq;   // DNA sequence
    private final int len;      // len of DNA sequence
    
    // creates a DNA object with the given name and sequence
    public DNA(String name, String sequence) {

        // Exception: empty name
        if (name.equals(""))
            throw new IllegalArgumentException("Empty string for given name");

        // Exception: not one of A, C, G, T
        for (int i = 0; i < sequence.length(); i++)
            if (!((sequence.charAt(i) == 'A') ||
                  (sequence.charAt(i) == 'C') ||
                  (sequence.charAt(i) == 'G') ||
                  (sequence.charAt(i) == 'T')))
                throw new IllegalArgumentException("Sequence: " + sequence +
                                                   " contains illegal char " +
                                                   sequence.charAt(i) + " @ " + i);
        // save in instance variables
        this.name = name;
        this.seq = sequence;
        this.len = sequence.length();
    }

    // returns the string representation of a DNA object
    public String toString() {
        String result = name + ": ";
        result += seq + " (" + len + "nt)";
        return result;
    }
    
    // total number of nucleotides in the sequence
    public int length() {
        return len;
    }

    // returns the DNA sequence string
    public String sequence() {
        return seq;
    }

    // returns true if this DNA sequence encodes a valid gene 
    // Based on code in Lecture: Using Data Types
    public boolean isGene() {

        //  Start codon (ATG)
        String[] START_CODON = {"ATG", "GTG"};

        // Stop codons (TAA, TAG, TGA) 
        String[] STOP_CODON = {"TAA", "TAG", "TGA"};

        // based on Using Data Types lecture (pp 16)

        // divisible by 3
        if (len % 3 != 0) return false;

        // check if start codon is ATG or GTG
        if ((!seq.startsWith(START_CODON[0])) &&
            (!seq.startsWith(START_CODON[1]))) return false;
                                
        // check  no stop codon is present in the sequence 
        // other than at the end of the sequence.
        for (int i = 3; i < seq.length() - 3; i += 3) {
            String codon = seq.substring(i, i + 3);
            for (int j = 0; j < STOP_CODON.length; j++) 
                if (codon.equals(STOP_CODON[j])) return false;
        }

        // sequence ends with one of the stop codons: either TAG, TAA, or TGA.
            for (int j = 0; j < STOP_CODON.length; j++) 
                if (seq.endsWith(STOP_CODON[j])) return true;

        // otherwise not a gene
        return false;
    }


    public int distanceTo(DNA other) {
        // throw exception if not the same length
        if (this.len != other.len)
            throw new IllegalArgumentException("Cannot compute hamming distance" +
                                               " since lengths are not the same: " +
                                               len + " : " + other.len);
        
        // go character by character
        int sum = 0;
        for (int i = 0; i < len; i++)
            if (seq.charAt(i) != other.seq.charAt(i))
                sum++;
        return sum;
    }

    // unit test
    public static void main(String[] args) {

        // test a gene and methods
        DNA genex = new DNA("geneX", "ATGCCCTAGTAA");
        StdOut.println(genex);
        StdOut.println(genex.length());
        StdOut.println(genex.sequence());
        StdOut.println(genex.isGene());

        // more tests
        DNA g1 = new DNA("g1", "ATGCATAGCGCATAG");
        StdOut.println(g1.isGene());
        DNA g2 = new DNA("g2", "GTGCTGCTGTGA");
        StdOut.println(g2.isGene());
        DNA g3 = new DNA("g3", "ATGCGCTGCGTCTGTACTAG");
        StdOut.println(g3.isGene());
        DNA g4 = new DNA("43", "ATGCCGTGACGTCTGTACTAG");
        StdOut.println(g4.isGene());

        // and distanceTo
        DNA c1 = new DNA("c1", "GGCATTA");
        StdOut.println(c1);
        StdOut.println(c1.distanceTo(c1));

        DNA d = new DNA("d", "ATGCATAGCGCATAG");
        StdOut.println(d.isGene());
        DNA e = new DNA("e", "ATGCATAGCGCATAA");
        StdOut.println(e.isGene());
    }
}
