// Check that number of calls to specified methods are within the specified min/max limits

// BUG: can't tell the difference between static method call Integer.parseInt
//      and instance method call (e.g., if Integer is the name of a reference variable)

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import java.util.TreeMap;

public class MethodCallCountCheck extends AbstractCheck {
    private int min = 0;                          // minimum number required
    private int max = Integer.MAX_VALUE;          // maximum number allowed
    private String[] methodNames = { };           // the list of method call names
    private TreeMap<String, Integer> counts;      // number of occurrences of each method call
    private int cumulativeCount = 0;              // total number of occurrences of all method calls
    private boolean sumMethodCallCounts = false;  // sum the method call counts?
    private String allMethodNames = "";           // set of names of all method calls to count, as a string
    private String callingMethodPattern = null;   // the calling method regexp to match (default = everything)

    /**
     * Keys pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_MIN = "method.call.count.min";
    public static final String MSG_MAX = "method.call.count.max";
    public static final String MSG_MAX_CUMULATIVE = "method.call.count.cumulative.max";
    public static final String MSG_MIN_CUMULATIVE = "method.call.count.cumulative.min";

    @Override
    public int[] getDefaultTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] { TokenTypes.METHOD_CALL, TokenTypes.LITERAL_NEW };
    }


    /**
      * Sets a maximum count.
      * @param max the maximum count.
      */
    public void setMax(int max) {
        this.max = max;
    }

    /**
      * Sets a minimum count.
      * @param min the minimum count.
      */
    public void setMin(int min) {
        this.min = min;
    }

    /**
     * Sets whether to use the sum of the method call counts,
     * rather than the individual method call counts.
     * @param sum whether to use the sum.
     */
    public void setSumMethodCallCounts(boolean sum) {
        sumMethodCallCounts = sum;
    }

    /**
     * Set the list of method names to count calls.
     * @param from array of method names to count calls
     */
    public final void setMethodNames(String... methodNames) {
        this.methodNames = methodNames.clone();
        allMethodNames = "{ ";
        for (int i = 0; i < methodNames.length - 1; i++) {
            allMethodNames += methodNames[i] + "(), ";
        }
        if (methodNames.length > 0)
            allMethodNames += methodNames[methodNames.length - 1] + "() ";
        allMethodNames += "}";
    }

    /**
     * Set the name of the calling method
     * @param callingMethodPattern the regexp for the calling method
     */
    public final void setCallingMethodPattern(String callingMethodPattern) {
        this.callingMethodPattern = callingMethodPattern;
    }


    /**
     * Returns whether a node is contained in the calling method.
     *
     * @param ast the node to check
     * @return a {@code boolean} value
     */
    public boolean isInCallingMethod(DetailAST ast) {
        if (callingMethodPattern == null) return true;
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF || x.getType() == TokenTypes.CTOR_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                return ident.getText().matches(callingMethodPattern);
            }
        }
        return false;
    }


    @Override
    public void visitToken(DetailAST ast) {
        // don't bother unless we're in the calling method (or no calling method is specified)
        if (!isInCallingMethod(ast)) return;

        String calledMethodName = null;

        // could be Integer.parseInt or p.distanceTo
        if (ast.findFirstToken(TokenTypes.DOT) != null) {
            DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
            FullIdent fullIdent = FullIdent.createFullIdent(dot);
            DetailAST shortIdent = Utilities.findLastToken(dot, TokenTypes.IDENT);
            String fullMethodName = fullIdent.getText();
            String shortMethodName = shortIdent.getText();
            if (counts.containsKey(fullMethodName)) {
                calledMethodName = fullMethodName;
            }
            else if (counts.containsKey(shortMethodName)) {
                calledMethodName = shortMethodName;
            }
        }

        // calling a static or instance method in the same class, or a constructor
        else {
            DetailAST ident = ast.findFirstToken(TokenTypes.IDENT);

            // creating an array (not calling a constructor)
            if (ident == null) return;

            if (counts.containsKey(ident.getText())) {
                calledMethodName = ident.getText();
            }
        }

        // found a method that we are counting
        if (calledMethodName != null) {

            counts.put(calledMethodName, counts.get(calledMethodName) + 1);
            cumulativeCount++;

            // flag max violations at point of discovery
            int count = counts.get(calledMethodName);
            if (!sumMethodCallCounts && (count > this.max)) {
                log(ast.getLineNo(),
                    MSG_MAX,
                    calledMethodName + "()",
                    count,
                    max);
            }

            // flag max cumulative violations at point of discovery
            if (sumMethodCallCounts && (cumulativeCount > this.max)) {
                log(ast.getLineNo(),
                    MSG_MAX_CUMULATIVE,
                    allMethodNames,
                    cumulativeCount,
                    max);
            }

        }

    }

    @Override
    public void beginTree(DetailAST rootAST) {
        counts = new TreeMap<String, Integer>();
        for (String methodName : methodNames) {
            counts.put(methodName, 0);
        }
        cumulativeCount = 0;
    }

    // can't flag violations of min counts until entire tree is traversed
    @Override
    public void finishTree(DetailAST rootAST) {

        // check cumulative counts
        if (sumMethodCallCounts) {
            if (cumulativeCount < this.min) {
                log(rootAST.getLineNo(),
                    MSG_MIN_CUMULATIVE,
                    allMethodNames,
                    cumulativeCount,
                    min);
            }
        }

        // check individual counts
        else {
            for (String methodName : methodNames) {
                int count = counts.get(methodName);
                if (count < this.min) {
                    log(rootAST.getLineNo(),
                        MSG_MIN,
                        methodName + "()",
                        count,
                        min);
                }
            }
        }
    }
}
