%{
	open InfoSyntax
	open Batteries

	(* Given s (the span of the LET token), the VAR v, 
	the expression e1 and the expresstion e2, 
	constructs a new Let expression with a span from s to e2. *)
	let span_exp_let (s: Span.t) (v: Span.t * string) (e1: span_exp) (e2: span_exp) =
	let span = Span.extend s e2.info in
	let exp = Let (snd v, e1, e2) in
	construct_exp span exp 

  (* Given s (the span of the IF token), the expression e1,
	the expression e2 and the expresstion e3, 
	constructs a new If expression with a span from s to e3. *)
	let span_exp_if (s: Span.t) (e1: span_exp) (e2: span_exp) (e3: span_exp) =
	let span = Span.extend s e3.info in
	let exp = If (e1, e2, e3) in
	construct_exp span exp

	(* Given the expression e1, an operator op, and the expression e2,
	constructs a new Op expression with a span from e1 to e2. *)
	let span_exp_op (e: span_exp) (op: Syntax.operator) (e2: span_exp) =
	let span = Span.extend e.info e2.info in
	let exp = Op (e, op, e2) in
	construct_exp span exp

	(* Given a VAR token, constructs a new Var expression with the 
	span and string of the token. *)
	let span_exp_var (v: Span.t * Syntax.variable) =
	let span = fst v in
	let exp = Var (snd v) in
	construct_exp span exp

	(* Given an INT token, constructs a new Constant expression. *)
	let span_exp_int (i: Span.t * int) =
	let span = fst i in
	let exp = Constant (Int (snd i)) in
	construct_exp span exp

	(* Given s (the span of a TRUE or FALSE token) and a bool b, 
	constructs a new Constant expression. *)
	let span_exp_bool (s: Span.t) (b: bool) =
	let span = s in 
	let exp = Constant (Bool b) in
	construct_exp span exp

  let span_exp_cons (e1: span_exp) (e2: span_exp) =
	let span = Span.extend e1.info e2.info in
	let exp = Cons (e1, e2) in
	construct_exp span exp

  let span_exp_match (s: Span.t) (e1: span_exp) b =
  let (e2, v1, v2, e3) = b in
	let span = Span.extend s e3.info in
	let exp = Match (e1, e2, snd(v1), snd(v2), e3) in
	construct_exp span exp

	let span_exp_rec (s: Span.t) d1 v  v2 (e1: span_exp) (e2: span_exp)=
	let span = Span.extend s e2.info in
  let exp1 = construct_exp span (Rec (d1, snd(v), snd(v2), e1)) in
  let exp = Let (snd(v), exp1, e2) in
	construct_exp span exp

  let span_exp_pair (e1: span_exp) (e2: span_exp) =
	let span = Span.extend e1.info e2.info in
	let exp = Pair (e1, e2) in
	construct_exp span exp

  let span_exp_fst (s: Span.t) (e1: span_exp) =
	let span = Span.extend s e1.info in
	let exp = Fst (e1) in
	construct_exp span exp

  let span_exp_snd (s: Span.t) (e1: span_exp) =
	let span = Span.extend s e1.info in
	let exp = Snd(e1) in
	construct_exp span exp

  let span_exp_app (e1: span_exp) (e2: span_exp) =
	let span = Span.extend e1.info e2.info in
	let exp = App (e1, e2) in
	construct_exp span exp



%}

%token <Span.t> TRUE
%token <Span.t> FALSE
%token <Span.t> EMPTY
%token <Span.t> IF
%token <Span.t> THEN
%token <Span.t> ELSE

%token <Span.t> MATCH
%token <Span.t> WITH
%token <Span.t> OR
%token <Span.t> TO
%token <Span.t> REC

%token <Span.t> FST
%token <Span.t> SND

%token <Span.t> EQ
%token <Span.t> LESS
%token <Span.t> LEQ

%token <Span.t * string> VAR
%token <Span.t> LET
%token <Span.t> IN
%token <Span.t> INT_T
%token <Span.t> BOOL_T
%token <Span.t> LIST
%token <Span.t> ARROW

%token <Span.t * int> INT
%token <Span.t> CONS
%token <Span.t> TIMES
%token <Span.t> PLUS
%token <Span.t> MINUS
%token <Span.t> COMMA

%token <Span.t> LPAREN
%token <Span.t> RPAREN

%token EOF

%start prog
%type  <InfoSyntax.span_exp> prog

/* Lowest precedence */
%right ELSE IN ARROW
%nonassoc LEQ LESS
%right CONS
%left PLUS MINUS
%left TIMES
/* Highest precedence */

%%

prog:
	| e = expr EOF                           { e }


expr:
  | sp=LET v=VAR EQ e1=expr IN e2=expr        { span_exp_let sp v e1 e2 }
  | sp=MATCH e1=expr WITH b=branches                    {span_exp_match sp e1 b}
  | sp=LET REC v=VAR LPAREN v2=VAR TO d1=datatype RPAREN TO d2=datatype EQ e1=expr IN e2=expr {span_exp_rec sp (Rec_T (d1, d2)) v v2 e1 e2}
  | sp=LPAREN e1=expr COMMA e2=expr RPAREN          { span_exp_pair e1 e2 }

  | sp=FST LPAREN e1=expr RPAREN                          { span_exp_fst sp e1 }
  | sp=SND LPAREN e1=expr RPAREN                           { span_exp_snd sp e1 }

	| sp=IF e1=expr THEN e2=expr ELSE e3=expr   { span_exp_if sp e1 e2 e3 }

  | e1=expr CONS e2=expr                   { span_exp_cons e1 e2 }
	| e1=expr TIMES e2=expr                     { span_exp_op e1 Times e2}
	| e1=expr PLUS e2=expr                      {  span_exp_op e1 Plus e2}
	| e1=expr MINUS e2=expr                     {  span_exp_op e1 Minus e2}

	| e1=expr LESS e2=expr                      { span_exp_op e1 Less e2}
	| e1=expr LEQ e2=expr                       {  span_exp_op e1 LessEq e2}

  | e2=expr2                                  {e2}

expr2:
    | e1=expr2 e2=expr3                       { span_exp_app e1 e2 }
    | e1=expr3                             { e1 }

expr3:
	| v=VAR                                     {span_exp_var v }
	| i=INT                                     { span_exp_int i }
  | sp=EMPTY                                   {construct_exp sp EmptyList }
	| sp=TRUE                                   { span_exp_bool sp true}
	| sp=FALSE                                  { span_exp_bool sp false }
  | LPAREN e=expr RPAREN                  { e }

datatype:
  | INT_T                                     {Int_T}
  | BOOL_T                                    {Bool_T}
  | d=datatype LIST                           {List_T (d)}
  | LPAREN d1=datatype ARROW d2=datatype RPAREN  {Rec_T (d1, d2)}
  | LPAREN d=datatype TIMES d2=datatype RPAREN   {Pair_T (d,d2)}

branches:
  |  OR EMPTY ARROW e1=expr OR v1=VAR CONS v2=VAR ARROW e2=expr{(e1, v1, v2, e2)}
  |  OR v1=VAR CONS v2=VAR ARROW e2=expr OR EMPTY ARROW e1=expr{(e1, v1, v2, e2)}

 ;
