{
	open Batteries
	open Parser
	open Span

	(* Returns a span based on the position of the current token in lexbuf. *)
	let position lexbuf =
		let start_pos = lexbuf.Lexing.lex_start_p in
		let end_pos = lexbuf.Lexing.lex_curr_p in
		{fname = start_pos.pos_fname; 
			start_line = start_pos.pos_lnum;
			end_line = end_pos.pos_lnum; 
			start_char = (Lexing.lexeme_start lexbuf) - start_pos.pos_bol+1; 
			end_char = (Lexing.lexeme_end lexbuf) - end_pos.pos_bol}

	(* Returns a pair of ints where the first int is the starting line and 
	the second int is the starting char of the current comment token.*)
	let comment_pos lexbuf : int*int =
		let start = lexbuf.Lexing.lex_curr_p in
		let sline = start.pos_lnum in
		let schar = (Lexing.lexeme_start lexbuf) - start.pos_bol+1
		in (sline, schar)

	(* Increments the line counter of lexbuf. *)
	let incr_linenum lexbuf =
		let pos = lexbuf.Lexing.lex_curr_p in
		lexbuf.Lexing.lex_curr_p <-
		{ pos with Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
			Lexing.pos_bol = pos.Lexing.pos_cnum; } ;;

}

let id = ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '_' '0'-'9']*
let num = ['0'-'9']+
let wspace = [' ' '\t']

rule token = parse
	| "(*"				{ comments 0 [comment_pos lexbuf] lexbuf }

	| "false"			{ FALSE (position lexbuf) }
	| "true"			{ TRUE (position lexbuf) }

	| "[]"			{ EMPTY (position lexbuf) }

	| "let"				{ LET (position lexbuf) }
	| "in"				{ IN (position lexbuf) }

	| "match"				{ MATCH (position lexbuf) }
  | "with"				{ WITH (position lexbuf) }
  | ":"				{ TO (position lexbuf) }
	| "rec"				{ REC (position lexbuf) }

	| "Fst"				{ FST (position lexbuf) }
	| "Snd"				{ SND (position lexbuf) }

	| "if"				{ IF (position lexbuf) }
	| "then"			{ THEN (position lexbuf) }
	| "else"			{ ELSE (position lexbuf) }
  | "int"			{ INT_T (position lexbuf) }
	| "bool"			{ BOOL_T (position lexbuf) }
  | "list"     	{ LIST (position lexbuf) }
  | "->"     	{ ARROW (position lexbuf) }

	| id as s			{ VAR (position lexbuf, s) }
	| num as n			{ INT (position lexbuf, int_of_string n) }


  | "|"				{ OR (position lexbuf) }
  | "::"				{ CONS (position lexbuf) }
	| "*"				{ TIMES (position lexbuf) }
	| "+"				{ PLUS (position lexbuf) }
	| "-"				{ MINUS (position lexbuf) }

	| ","				{ COMMA (position lexbuf) }

	| "<="				{ LEQ (position lexbuf) }

	| "<"				{ LESS (position lexbuf) }
	| "="				{ EQ (position lexbuf) }

	| "("				{ LPAREN (position lexbuf) }
	| ")"				{ RPAREN (position lexbuf) }

	| wspace			{ token lexbuf }
	| '\n'				{ incr_linenum lexbuf; token lexbuf}
	| _					{ token lexbuf }
	| eof 				{ EOF }

and comments level positions = parse
  | "*)"  { if level = 0 then token lexbuf 
			else 
			  	(match positions with 
			  	| hd::tl -> comments (level-1) tl lexbuf
			  	| _ -> Error.error()) }
  | "(*"  { comments (level+1) ((comment_pos lexbuf)::positions) lexbuf }
  | '\n'  { incr_linenum lexbuf; comments level positions lexbuf}
  | _     { comments level positions lexbuf }
  | eof   { match positions with
			| hd::tl -> Error.eof_error hd
			| _ -> Error.error() }


