open Batteries
open InfoSyntax

(* Parses the file specified by the string filename and 
returns an span_exp. *)
let read ?(filename : string option = None) lexbuf : span_exp =
    let get_info () =
        let curr = lexbuf.Lexing.lex_curr_p in
        let line = curr.Lexing.pos_lnum in
        let cnum = curr.Lexing.pos_cnum - curr.Lexing.pos_bol in
        let tok = Lexing.lexeme lexbuf in
        (tok, line, cnum)
    in
    let header =
        match filename with
        | None -> "[Parser Error]"
        | Some s -> Printf.sprintf "[Parser Error in %s]" s
    in
    try Parser.prog Lexer.token lexbuf with
    | Error.EOF position->
        let msg = 
            Printf.sprintf "%s End of file found in the comment starting on %s." 
            header position
        in Error.error_msg msg
    | _ ->
        let (tok,line,cnum) = get_info () in
        let msg =
            Printf.sprintf "%s Unexpected token \"%s\" in line: %s, char: %s." 
            header tok (string_of_int line) (string_of_int cnum)
        in Error.error_msg msg

(* Reads and parses the information from the 
input channel cin. Returns a span_exp. *)
let read_from_in cin : span_exp =
    let res = read (Lexing.from_channel cin) in
    close_in cin ; res

(* Reads and parses the string str. Returns a span_exp.*)
let read_from_str (str : string) : span_exp = 
    Lexing.from_string str |> read

(* Reads and parses from the file fname. Returns a span_exp. *)
let read_from_file (fname : string) : span_exp=
    let fin = open_in fname in
    let lexbuf = Lexing.from_channel fin in
    lexbuf.lex_curr_p <- {lexbuf.lex_curr_p with pos_fname=fname};
    lexbuf.lex_start_p <- {lexbuf.lex_start_p with pos_fname=fname};
    let res = read ~filename:(Some fname) lexbuf in
    close_in fin ; res
