open Syntax

type datatype =
  | Int_T
  | Bool_T
  | List_T of datatype
  | Pair_T of datatype * datatype
  | Rec_T of datatype * datatype
  | Temp

type 'info e =
| Var of Syntax.variable   
  | Constant of Syntax.constant
  | Op of 'info exp * Syntax.operator * 'info exp
  (* If (conditional, consequent, alternative) *)
  | If of 'info exp * 'info exp * 'info exp
  (* Let (v, e1, e2) is a let binding with the form let v = e1 in e2 *)
  | Let of Syntax.variable * 'info exp * 'info exp

  (* Pairs *)
  | Pair of 'info exp * 'info exp
  | Fst of 'info exp
  | Snd of 'info exp

  (* Lists *)
  | EmptyList
  (* Note 1: this permits type-heterogeneity, unlike OCaml, e.g.:
     Cons (Constant (Int 1), Cons (Constant (Bool true), EmptyList)) 
    
     Note 2: this also doesn't enforce that a list should end in a 
     Cons with EmptyList as the second argument, but we will only 
     ever use well-defined lists of this structure.
  *)
  | Cons of 'info exp * 'info exp
  (*   Match (e1, e2, hd, tl, e3) is a match statement with the form:
   match e1 with 
   | [] -> e2 
   | hd::tl -> e3 
  *)                  
  | Match of 'info exp * 'info exp * Syntax.variable * Syntax.variable * 'info exp  

  (* Functions *)
  (* Rec (f, x, b) is the definition of a possibly-recursive function
     The function is named f and x is the name of the parameter. b is 
     the body of the expression, and may contain f and/or x.
  *)
  | Rec of datatype * Syntax.variable * Syntax.variable * 'info exp

  (* App (f, x) is a function call of closure f with argument x *)
  | App of 'info exp * 'info exp
and 'info exp = { info:'info; e:'info e }

type span_exp = Span.t exp

(* Constructor function to create 'info exp. *)
val construct_exp : 'info -> 'info e -> 'info exp

type 'info env = (Syntax.variable * 'info exp) list

val empty_env : 'info env

(* If the variable has an associated 'info exp in the env, then 
returns the associated 'info exp as an option. Otherwise, 
returns None. *)
val lookup_env : 'info env -> variable -> ('info exp) option

(* Returns a new 'info env containing the variable and its
associated 'info exp. *)
val update_env : 'info env -> variable -> 'info exp -> 'info env

(* Coverts the given 'info exp to a unit exp *)
val to_exp : 'info exp -> Syntax.exp
