open Syntax

let var_eq x y = (String.compare x y = 0)
let var_neq x y = not (String.compare x y = 0)


(*Datatyoes for typechecking*)
type datatype =
  | Int_T
  | Bool_T
  | List_T of datatype
  | Pair_T of datatype * datatype
  | Rec_T of datatype * datatype
  | Temp

(*Internal data type for typechecking*)
type 'info e =
     (* Basic *)
  | Var of Syntax.variable   
  | Constant of Syntax.constant
  | Op of 'info exp * Syntax.operator * 'info exp
  (* If (conditional, consequent, alternative) *)
  | If of 'info exp * 'info exp * 'info exp
  (* Let (v, e1, e2) is a let binding with the form let v = e1 in e2 *)
  | Let of Syntax.variable * 'info exp * 'info exp

  (* Pairs *)
  | Pair of 'info exp * 'info exp
  | Fst of 'info exp
  | Snd of 'info exp

  (* Lists *)
  | EmptyList
  (* Note 1: this permits type-heterogeneity, unlike OCaml, e.g.:
     Cons (Constant (Int 1), Cons (Constant (Bool true), EmptyList)) 
    
     Note 2: this also doesn't enforce that a list should end in a 
     Cons with EmptyList as the second argument, but we will only 
     ever use well-defined lists of this structure.
  *)
  | Cons of 'info exp * 'info exp
  (*   Match (e1, e2, hd, tl, e3) is a match statement with the form:
   match e1 with 
   | [] -> e2 
   | hd::tl -> e3 
  *)                  
  | Match of 'info exp * 'info exp * Syntax.variable * Syntax.variable * 'info exp  

  (* Functions *)
  (* Rec (f, x, b) is the definition of a possibly-recursive function
     The function is named f and x is the name of the parameter. b is 
     the body of the expression, and may contain f and/or x.
  *)
  | Rec of datatype * Syntax.variable * Syntax.variable * 'info exp

  (* App (f, x) is a function call of closure f with argument x *)
  | App of 'info exp * 'info exp
and 'info exp = { info:'info; e:'info e }

type span_exp = Span.t exp

type 'info env = (Syntax.variable * 'info exp) list

let empty_env : 'info env = []

(* Constructor function to create exp. *)
let construct_exp (info:'info) (e: 'info e) =
	{ info=info; e=e }

(* If the x has an associated exp in the env, then 
returns the associated exp as an option. Otherwise, 
returns None. *)
let rec lookup_env (env:'info env) (x:Syntax.variable) : ('info exp) option =
    match env with 
    | [] -> None
    | (xe, ve)::tl -> 
        if (var_eq xe x) then Some ve else lookup_env tl x

(* Returns a new env containing the pair (x,v). *)
let update_env (env:'info env) (x:Syntax.variable) (v:'info exp) : 'info env = 
    let rec loop (e: 'info env) (var:Syntax.variable) (value:'info exp) =
        match e with
        | [] -> [(var, value)]
        | (var2, value2)::tl -> 
            if (var_eq var var2) then (var, value)::tl
            else (var2,value2)::(loop tl var value)
    in
    loop env x v



(* Coverts e to a exp *)
let rec to_exp (e: 'info exp) : Syntax.exp =
  match e.e with
  | Var v -> Syntax.Var v
  | Constant c -> Syntax.Constant c
  | Op (e, o, e2) -> Syntax.Op(to_exp(e), o, to_exp(e2))
  (* If (conditional, consequent, alternative) *)
  | If (e1, e2, e3) -> Syntax.If(to_exp(e1), to_exp(e2), to_exp(e3))
  (* Let (v, e1, e2) is a let binding with the form let v = e1 in e2 *)
  | Let (v, e1, e2) -> Syntax.Let(v, to_exp(e1), to_exp(e2))

  (* Pairs *)
  | Pair (e1, e2) -> Syntax.Pair(to_exp(e1), to_exp(e2))
  | Fst(e1) -> Syntax.Fst(to_exp(e1))
  | Snd(e1) -> Syntax.Snd(to_exp(e1))

  (* Lists *)
  | EmptyList -> Syntax.EmptyList
  | Cons (e1, e2) -> Syntax.Cons(to_exp(e1), to_exp(e2))
  | Match (e1, e2, v1, v2, e3) -> Syntax.Match(to_exp(e1), to_exp(e2), v1, v2, to_exp(e3))

  (* Functions *)
  (* Rec (f, x, b) is the definition of a possibly-recursive function
     The function is named f and x is the name of the parameter. b is 
     the body of the expression, and may contain f and/or x.
  *)
  | Rec (d1, v1, v2, e1) -> Syntax.Rec(v1, v2, to_exp(e1))

  (* App (f, x) is a function call of closure f with argument x *)
  | App (e1, e2) -> Syntax.App(to_exp(e1), to_exp(e2)) 

