open Format 

exception Error
exception EOF of string
exception DescriptiveError of string

(* Raises a general error. *)
let error () =
	raise (Error)

(* Raises a descriptive error with msg *)
let error_msg (msg:string) =
	raise (DescriptiveError msg)

(* Raises an EOF error with pos (a pair of ints where
the first int is the starting line and the second int
is the starting char of the non-terminating comment). *)
let eof_error (pos:int*int) =
	let (sline, schar) = pos in
	let str = 
		Printf.sprintf "line: %d, char: %d" sline schar
	in raise (EOF str)
	
(* Raises a descriptive error based on the given strings and span:
ty is the type that makes the expression typecheck, bad_ty is 
the incorrect type that prevents proper typechecking, exp is the string 
representation of the erroneous expression, and span is the original
location of the erroneous expression. *)
let type_error (ty: string) (bad_ty: string) (exp: string) (span: Span.t) =
	let str_span = Span.string_of_span span in
	let msg = 
		Printf.sprintf "[Type Error in %s] The expression \"%s\" has type %s but was expected to have type %s."
		str_span exp bad_ty ty
	in raise (DescriptiveError msg)

(* Raises a descriptive error where v is the unbound variable and
span is the original location of the unbound variable. *)
let unbound_error (v: string) (span: Span.t) =
	let str_span = Span.string_of_span span in
	let msg = Printf.sprintf "[Error in %s] Unbound variable %s." str_span v 
	in raise (DescriptiveError msg)

(* Raises a descriptive error where span is the original location of 
the bad operator. *)
let op_error (span: Span.t) =
	let str_span = Span.string_of_span span in
	let msg = Printf.sprintf "[Error in %s] Bad Operator." str_span 
	in raise (DescriptiveError msg)

(* Uses the pretty-printing module to properly print out msg. *)
let print_error_msg (msg: string) =
  open_hovbox 0; pp_print_text std_formatter msg; close_box ();
