// BUG: fully qualified names for methods aren't the only way to refer to a method
//      can't tell the difference between static method call Integer.parseInt
//      and instance method call (e.g., if Integer is the name of a reference variable)
//      overloading, promotion, subtyping, checkstyle is not type aware
//      https://github.com/checkstyle/checkstyle/issues/535

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;

public class SideEffectCheck extends AbstractCheck {

    /** The regexp for standard input methods. */
    private String formatStdOut       = "(System\\.out|System\\.err|StdOut)\\.(print|printf|println|write)";
    private String formatStdIn        = "System.in.read|StdIn\\..*";
    private String formatBinaryStdOut = "BinaryStdOut\\..*";
    private String formatBinaryStdIn  = "BinaryStdIn\\..*";
    private String formatStdDraw      = "StdDraw\\..*";
    private String formatStdAudio     = "StdAudio\\..*";

    private boolean prohibitStdOut       = false;
    private boolean prohibitStdIn        = false;
    private boolean prohibitBinaryStdOut = false;
    private boolean prohibitBinaryStdIn  = false;
    private boolean prohibitStdDraw      = false;
    private boolean prohibitStdAudio     = false;
    private boolean ignoreMain           = false;

    /** The regexp for the calling method (null means ban all method calls regardless of context). */
    private String callingMethodPattern = null;

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_STDOUT             = "side.effect.stdout";
    public static final String MSG_STDOUT_FROM        = "side.effect.stdout.from";
    public static final String MSG_STDIN              = "side.effect.stdin";
    public static final String MSG_STDIN_FROM         = "side.effect.stdin.from";
    public static final String MSG_BINARY_STDOUT      = "side.effect.binarystdout";
    public static final String MSG_BINARY_STDOUT_FROM = "side.effect.binarystdout.from";
    public static final String MSG_BINARY_STDIN       = "side.effect.binarystdin";
    public static final String MSG_BINARY_STDIN_FROM  = "side.effect.binarystdin.from";
    public static final String MSG_STDDRAW            = "side.effect.stddraw";
    public static final String MSG_STDDRAW_FROM       = "side.effect.stddraw.from";
    public static final String MSG_STDAUDIO           = "side.effect.stdaudio";
    public static final String MSG_STDAUDIO_FROM      = "side.effect.stdaudio.from";

    @Override
    public int[] getDefaultTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }

    /**
     * Set whether to prohibit printing to standard input
     * @param prohibit decide whether to prohibit printing to standard output
     */
    public void setProhibitStdOut(boolean prohibit) {
        prohibitStdOut = prohibit;
    }

    /**
     * Set whether to prohibit reading from standard input
     * @param prohibit decide whether to prohibit reading from standard input
     */
    public void setProhibitStdIn(boolean prohibit) {
        prohibitStdIn = prohibit;
    }

    /**
     * Set whether to prohibit printing to binary standard input
     * @param prohibit decide whether to prohibit printing to binary standard output
     */
    public void setProhibitBinaryStdOut(boolean prohibit) {
        prohibitBinaryStdOut = prohibit;
    }

    /**
     * Set whether to prohibit reading from binary standard input
     * @param prohibit decide whether to prohibit reading from binary standard input
     */
    public void setProhibitBinaryStdIn(boolean prohibit) {
        prohibitBinaryStdIn = prohibit;
    }

    /**
     * Set whether to prohibit drawing to standard drawing
     * @param prohibit decide whether to prohibit drawing to standard drawing
     */
    public void setProhibitStdDraw(boolean prohibit) {
        prohibitStdDraw = prohibit;
    }

    /**
     * Set whether to prohibit playing on standard audio
     * @param prohibit decide whether to prohibit playing on standard audio
     */
    public void setProhibitStdAudio(boolean prohibit) {
        prohibitStdAudio = prohibit;
    }

    /**
     * Set whether to ignore main()
     * @param ignore side effects in main()
     */
    public void setIgnoreMain(boolean ignore) {
        ignoreMain = ignore;
    }


    /**
     * Set the name of the calling method
     * @param callingMethodPattern the regexp for the calling method
     */
    public final void setCallingMethodPattern(String callingMethodPattern) {
        this.callingMethodPattern = callingMethodPattern;
    }

    // returns whether a node is contained in a calling method
    private boolean isInCallingMethod(DetailAST ast) {

        // check to ignore main()
        if (ignoreMain && Utilities.isInMainMethod(ast)) {
            return false;
        }

        // check if calling method pattern
        if (callingMethodPattern == null) return true;

        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF || x.getType() == TokenTypes.CTOR_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                return ident.getText().matches(callingMethodPattern);
            }
        }
        return false;
    }

    // returns the name of the calling method
    private String getCallingMethodName(DetailAST ast) {
        for (DetailAST x = ast; x != null; x = x.getParent()) {
            if (x.getType() == TokenTypes.METHOD_DEF || x.getType() == TokenTypes.CTOR_DEF) {
                DetailAST ident = x.findFirstToken(TokenTypes.IDENT);
                // if (ident.getText().matches(callingMethodPattern))
                    return ident.getText();
            }
        }
        return null;
    }

    @Override
    public void visitToken(DetailAST ast) {

        // don't bother unless we're in the calling method (or no calling method is specified)
        if (!isInCallingMethod(ast)) return;
 
        // could be Integer.parseInt or p.distanceTo
        if (ast.findFirstToken(TokenTypes.DOT) != null) {
            DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
            FullIdent fullIdent = FullIdent.createFullIdent(dot);
            DetailAST shortIdent = Utilities.findLastToken(dot, TokenTypes.IDENT);
            String fullMethodName = fullIdent.getText();
            if (prohibitStdOut && fullMethodName.matches(formatStdOut)) {
                reportErrorStdOut(ast, fullMethodName);
            }
            else if (prohibitStdIn && fullMethodName.matches(formatStdIn)) {
                reportErrorStdIn(ast, fullMethodName);
            }
            else if (prohibitBinaryStdOut && fullMethodName.matches(formatBinaryStdOut)) {
                reportErrorBinaryStdOut(ast, fullMethodName);
            }
            else if (prohibitBinaryStdIn && fullMethodName.matches(formatBinaryStdIn)) {
                reportErrorBinaryStdIn(ast, fullMethodName);
            }
            else if (prohibitStdDraw && fullMethodName.matches(formatStdDraw)) {
                reportErrorStdDraw(ast, fullMethodName);
            }
            else if (prohibitStdAudio && fullMethodName.matches(formatStdAudio)) {
                reportErrorStdAudio(ast, fullMethodName);
            }
        }
    }

    private void reportErrorStdIn(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDIN,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDIN_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }

    private void reportErrorStdOut(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDOUT,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDOUT_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }

    private void reportErrorBinaryStdIn(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_BINARY_STDIN,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_BINARY_STDIN_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }

    private void reportErrorBinaryStdOut(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_BINARY_STDOUT,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_BINARY_STDOUT_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }

    private void reportErrorStdDraw(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDDRAW,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDDRAW_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }

    private void reportErrorStdAudio(DetailAST ast, String calledMethodName) {
        if (callingMethodPattern == null && !ignoreMain) {
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDAUDIO,
                calledMethodName + "()");
        }

        else {
            String callingMethodName = getCallingMethodName(ast);
            log(ast.getLineNo(),
                ast.getColumnNo(),
                MSG_STDAUDIO_FROM,
                calledMethodName + "()",
                callingMethodName + "()");
        }
    }
}
