// not-so-ideal feature: if a student comments two instance variables on one line,
//                       then this test will flag the second variable as uncommmented unless it appears
//                       on the same line as the first

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;

import java.util.Collections;
import java.util.List;
import java.util.TreeSet;
import java.util.Set;

public class CommentCheck extends AbstractCheck {

    // If true, don't apply the check to static final variables (constants)
    private boolean ignoreStaticFinalVariables = false;  

    // line numbers containing single-line comments
    private Set<Integer> singleLineCommentLineNumbers;

    // line numbers containing block comments (delimited by /* and */)
    private Set<Integer> blockCommentLineNumbers;

    // line numbers containing code
    private Set<Integer> codeLineNumbers;

    // set of method names to ignore
    private final Set<String> ignoredMethodNames = new TreeSet<>();


    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_INSTANCE_VARIABLE = "comment.instance.variable";
    public static final String MSG_STATIC_VARIABLE = "comment.static.variable";
    public static final String MSG_INSTANCE_METHOD = "comment.instance.method";
    public static final String MSG_STATIC_METHOD = "comment.static.method";
    public static final String MSG_CONSTRUCTOR = "comment.constructor";

    @Override
    public int[] getDefaultTokens() {
        return new int[] {
            TokenTypes.VARIABLE_DEF,
            TokenTypes.METHOD_DEF,
            TokenTypes.CTOR_DEF
        };
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] { };
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {
            TokenTypes.VARIABLE_DEF,
            TokenTypes.METHOD_DEF,
            TokenTypes.CTOR_DEF
        };
    }

    /**
     * Set the list of ignore method names.
     * @param methodNames array of ignored method names
     */
    public void setIgnoredMethodNames(String... methodNames) {
        ignoredMethodNames.clear();
  	Collections.addAll(ignoredMethodNames, methodNames);
    }

   /**
     * Sets whether we should apply the check to the static final variables (constants).
     *
     * @param ignore new value of the property.
     */
    public void setIgnoreStaticFinalVariables(boolean ignore) {
        ignoreStaticFinalVariables = ignore;
    }

    @Override
    public void beginTree(DetailAST rootAST) {

        FileContents fileContents = getFileContents();

        // line numbers containing single-line comments (those that begin with //)
        singleLineCommentLineNumbers = new TreeSet<Integer>();
        for (int lineNumber : fileContents.getSingleLineComments().keySet()) {
            singleLineCommentLineNumbers.add(lineNumber);
        }

        // line numbers containing block comments (those delimited by /* and */)
        blockCommentLineNumbers = new TreeSet<Integer>();
        for (List<TextBlock> list : fileContents.getBlockComments().values()) {
            for (TextBlock block : list) {
                for (int i = block.getStartLineNo(); i <= block.getEndLineNo(); i++)
                    blockCommentLineNumbers.add(i);
            }
        }

        // line numbers containing code that is not a comment
        codeLineNumbers = new TreeSet<Integer>();
        for (DetailAST sibling = rootAST; sibling != null; sibling = sibling.getNextSibling())
            identifyLineNumbersContainingCode(sibling);
    }

    // find all line numbers containing code (other than comments)
    private void identifyLineNumbersContainingCode(DetailAST ast) {
        int lineNo = ast.getLineNo();
        codeLineNumbers.add(lineNo);
        for (DetailAST child = ast.getFirstChild(); child != null; child = child.getNextSibling())
            identifyLineNumbersContainingCode(child);
    }


    @Override
    public void visitToken(DetailAST ast) {

/*
        // not in an outermost class
        if (!Utilities.isInOuterMostClass(ast)) {
            return;
        }
*/

        if (ast.getType() == TokenTypes.VARIABLE_DEF) {
            if (ignoreStaticFinalVariables && Utilities.isStaticFinalVariable(ast))
                return;
            if ((Utilities.isInstanceVariable(ast) || Utilities.isStaticVariable(ast)) && !hasComment(ast)) {
                reportStyleError(ast);
            }
        }

        else if (ast.getType() == TokenTypes.METHOD_DEF) {
            String methodName = findName(ast);
            if (!ignoredMethodNames.contains(methodName) && !hasComment(ast)) {
                reportStyleError(ast);
            }
        }

        else if (ast.getType() == TokenTypes.CTOR_DEF) {
            String constructorName = findName(ast);
            if (!ignoredMethodNames.contains(constructorName) && !hasComment(ast)) {
                reportStyleError(ast);
            }
        }
    }

    // does the variable have an associated comment on the same line (for variable declarations)
    // (or previous line if no other code on that line)?
    private boolean hasComment(DetailAST ast) {
        FileContents fileContents = getFileContents();
        int lineNo = ast.getLineNo();

        // allow single-line comments on the same line for variable declarations
        if (ast.getType() == TokenTypes.VARIABLE_DEF) {
            if (singleLineCommentLineNumbers.contains(lineNo)) return true;
            if (blockCommentLineNumbers.contains(lineNo)) return true;
        }

        if (lineNo == 1) return false;
        if (codeLineNumbers.contains(lineNo - 1)) return false;
        if (blockCommentLineNumbers.contains(lineNo - 1)) return true;
        if (singleLineCommentLineNumbers.contains(lineNo - 1)) return true;
        return false;
    }

    private String findName(DetailAST ast) {
        DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);
        return identifier.getText();
    }

    private void reportStyleError(DetailAST ast) {
        String name = findName(ast);

        // use method or variable name when logging line and column numbers
        DetailAST identifier = ast.findFirstToken(TokenTypes.IDENT);


        if (Utilities.isStaticVariable(ast)) {
            log(identifier,
                MSG_STATIC_VARIABLE,
                name);
        }
        else if (Utilities.isInstanceVariable(ast)) {
            log(identifier,
                MSG_INSTANCE_VARIABLE,
                name);
        }

        else if (Utilities.isStaticMethod(ast)) {
            log(identifier,
                MSG_STATIC_METHOD,
                name + "()");
        }

        else if (Utilities.isInstanceMethod(ast)) {
            log(identifier,
                MSG_INSTANCE_METHOD,
                name + "()");
        }

        else if (Utilities.isConstructor(ast)) {
            log(identifier,
                MSG_CONSTRUCTOR,
                name + "()");
        }

        else throw new IllegalStateException("neither a class variable nor an instance variable");

    }

}
