// Checks that the number of calls to Integer.parseInt() and related methods is equal
// to the corresponding number of command-line arguments.

// TODO: only consider if it is inside main()
// TODO: only consider if argument is named args[i] ?

// BUG (minor): can't tell the difference between static method call Integer.parseInt
//      and instance method call (e.g., if Integer is the name of a reference variable)

package edu.princeton.cs.lift.checkstyle;

import com.puppycrawl.tools.checkstyle.api.*;
import java.util.TreeMap;

public class CommandLineArgumentCheck extends AbstractCheck {
    private int integerParseIntCount = 0;         // number of calls to Integer.parseInt()
    private int integerCommandLineArguments = 0;  // number of integer command-line arguments
    private int doubleParseDoubleCount = 0;       // number of calls to Double.parseDouble()
    private int doubleCommandLineArguments = 0;   // number of double command-line arguments

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_INT = "integer.parseInt";
    public static final String MSG_INT_ZERO = "integer.parseInt.zero";

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_DOUBLE = "double.parseDouble";
    public static final String MSG_DOUBLE_ZERO = "double.parseDouble.zero";


    @Override
    public int[] getDefaultTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }

    @Override
    public int[] getRequiredTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }

    @Override
    public int[] getAcceptableTokens() {
        return new int[] {TokenTypes.METHOD_CALL};
    }


    /**
      * Sets the number of int command-line arguments.
      * @param value the number of int command-line arguments.
      */
    public void setInt(int value) {
        this.integerCommandLineArguments = value;
    }

    /**
      * Sets the number of double command-line arguments.
      * @param value the number of double command-line arguments.
      */
    public void setDouble(int value) {
        this.doubleCommandLineArguments = value;
    }


    @Override
    public void visitToken(DetailAST ast) {
        if (ast.findFirstToken(TokenTypes.DOT) == null) return;
        DetailAST dot = ast.findFirstToken(TokenTypes.DOT);
        FullIdent fullIdent = FullIdent.createFullIdent(dot);
        String methodName = fullIdent.getText();
        if ("Integer.parseInt".equals(methodName)) {
            integerParseIntCount++;
            if (integerCommandLineArguments == 0) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_INT_ZERO);
            }
        }
        if ("Double.parseDouble".equals(methodName)) {
            doubleParseDoubleCount++;
            if (doubleCommandLineArguments == 0) {
                log(ast.getLineNo(),
                    ast.getColumnNo(),
                    MSG_DOUBLE_ZERO);
            }
        }
    }

    @Override
    public void beginTree(DetailAST rootAST) {
        integerParseIntCount = 0;
        doubleParseDoubleCount = 0;
    }

    @Override
    public void finishTree(DetailAST rootAST) {
        if ((integerCommandLineArguments > 0) && (integerParseIntCount != integerCommandLineArguments)) {
            log(rootAST.getLineNo(),
                MSG_INT,
                integerParseIntCount,
                integerCommandLineArguments);
        }
        if ((doubleCommandLineArguments > 0) && (doubleParseDoubleCount != doubleCommandLineArguments)) {
            log(rootAST.getLineNo(),
                MSG_DOUBLE,
                doubleParseDoubleCount,
                doubleCommandLineArguments);
        }
    }
}
