// Source file for GAPS scalar grid class



////////////////////////////////////////////////////////////////////////
// NOTE:
// Grid values are defined as samples at the grid positions ranging from
// (0, 0, 0) to (xres-1, yres-1).  Grid values outside this range
// are undefined.
////////////////////////////////////////////////////////////////////////



// Include files

#include "R2Shapes/R2Shapes.h"



R2Grid::
R2Grid(int xresolution, int yresolution)
{
  // Set grid resolution
  grid_resolution[0] = xresolution;
  grid_resolution[1] = yresolution;
  grid_row_size = xresolution;
  grid_size = grid_row_size * yresolution;

  // Allocate grid values
  if (grid_size == 0) grid_values = NULL;
  else grid_values = new RNScalar [ grid_size ];
  assert(!grid_size || grid_values);

  // Set transformations
  grid_to_world_transform = R2identity_affine;
  world_to_grid_transform = R2identity_affine;
  world_to_grid_scale_factor = 1.0;

  // Set all values to zero
  Clear(0);
}



R2Grid::
R2Grid(const R2Grid& voxels)
  : grid_values(NULL)
{
  // Copy everything
  *this = voxels;
}



R2Grid::
~R2Grid(void)
{
  // Deallocate memory for grid values
  if (grid_values) delete [] grid_values;
}



RNInterval R2Grid::
Range(void) const
{
  // Find smallest and largest values
  RNScalar minimum = FLT_MAX;
  RNScalar maximum = -FLT_MAX;
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    if (*grid_valuep < minimum) minimum = *grid_valuep;
    if (*grid_valuep > maximum) maximum = *grid_valuep;
    grid_valuep++;
  }
  return RNInterval(minimum, maximum);
}



RNScalar R2Grid::
L1Norm(void) const
{
  // Return L1 norm of grid
  RNScalar sum = 0.0;
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    sum += *(grid_valuep++);
  return sum;
}



RNScalar R2Grid::
L2Norm(void) const
{
  // Return L2 norm of grid
  RNScalar sum = 0.0;
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    RNScalar value = *(grid_valuep++);
    sum += value * value;
  }
  return sqrt(sum);
}



int R2Grid::
Cardinality(void) const
{
  // Return number of non-zero grid values
  int count = 0;
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    if (*(grid_valuep++) != 0.0) count++;
  return count;
}



RNScalar R2Grid::
GridValue(RNScalar x, RNScalar y) const
{
  // Check if within bounds
  if ((x < 0) || (x > grid_resolution[0]-1)) return 0.0;
  if ((y < 0) || (y > grid_resolution[1]-1)) return 0.0;

  // Bilinear interpolation
  int ix1 = (int) x;
  int iy1 = (int) y;
  int ix2 = ix1 + 1;
  int iy2 = iy1 + 1;
  if (ix2 >= grid_resolution[0]) ix2 = ix1;
  if (iy2 >= grid_resolution[1]) iy2 = iy1;
  float dx = x - ix1;
  float dy = y - iy1;
  float value = 0.0;
  value += GridValue(ix1, iy1) * (1.0-dx) * (1.0-dy);
  value += GridValue(ix1, iy2) * (1.0-dx) * dy;
  value += GridValue(ix2, iy1) * dx * (1.0-dy);
  value += GridValue(ix2, iy2) * dx * dy;
  return value;
}



R2Grid& R2Grid::
operator=(const R2Grid& voxels) 
{
  // Copy grid resolution
  grid_resolution[0] = voxels.grid_resolution[0];
  grid_resolution[1] = voxels.grid_resolution[1];
  grid_row_size = voxels.grid_row_size;
  grid_size = voxels.grid_size;

  // Copy grid values
  if (grid_values) delete [] grid_values;
  if (grid_size == 0) grid_values = NULL;
  else grid_values = new RNScalar [ grid_size ];
  assert(!grid_size || grid_values);
  for (int i = 0; i < grid_size; i++) {
    grid_values[i] = voxels.grid_values[i];
  }

  // Copy transforms
  grid_to_world_transform = voxels.grid_to_world_transform;
  world_to_grid_transform = voxels.world_to_grid_transform;
  world_to_grid_scale_factor = voxels.world_to_grid_scale_factor;

  // Return this
  return *this;
}



void R2Grid::
Abs(void) 
{
  // Take absolute value of every grid value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    *grid_valuep = fabs(*(grid_valuep++));
}



void R2Grid::
Sqrt(void) 
{
  // Take sqrt of every grid value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    *grid_valuep = sqrt(*(grid_valuep++));
}



void R2Grid::
Square(void) 
{
  // Square every grid value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    *grid_valuep = (*grid_valuep) * (*grid_valuep);
    grid_valuep++;
  }
}



void R2Grid::
Negate(void) 
{
  // Negate every grid value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    *grid_valuep = -(*grid_valuep);
    grid_valuep++;
  }
}



void R2Grid::
Invert(void) 
{
  // Invert every grid value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    if (RNIsNotZero(*grid_valuep)) *grid_valuep = 1.0/(*grid_valuep);
    grid_valuep++;
  }
}



void R2Grid::
Normalize(void) 
{
  // Scale so that length of "vector" is one
  Divide(L2Norm());
}



void R2Grid::
Clear(RNScalar value) 
{
  // Set all grid values to value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    *(grid_valuep++) = value;
}



void R2Grid::
Add(RNScalar value) 
{
  // Add value to all grid values 
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    *(grid_valuep++) += value;
}



void R2Grid::
Add(const R2Grid& voxels) 
{
  // Resolutions must be the same (for now)
  assert(grid_resolution[0] == voxels.grid_resolution[0]);
  assert(grid_resolution[1] == voxels.grid_resolution[1]);

  // Add passed grid values to corresponding entries of this grid
  for (int i = 0; i < grid_size; i++) 
    grid_values[i] += voxels.grid_values[i];
}



void R2Grid::
Subtract(RNScalar value) 
{
  // Add the opposite
  Add(-value);
}



void R2Grid::
Subtract(const R2Grid& voxels) 
{
  // Resolutions must be the same (for now)
  assert(grid_resolution[0] == voxels.grid_resolution[0]);
  assert(grid_resolution[1] == voxels.grid_resolution[1]);

  // Subtract passed grid values from corresponding entries of this grid
  for (int i = 0; i < grid_size; i++) 
    grid_values[i] -= voxels.grid_values[i];
}



void R2Grid::
Multiply(RNScalar value) 
{
  // Multiply grid values by value
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) 
    *(grid_valuep++) *= value;
}



void R2Grid::
Multiply(const R2Grid& voxels) 
{
  // Resolutions must be the same (for now)
  assert(grid_resolution[0] == voxels.grid_resolution[0]);
  assert(grid_resolution[1] == voxels.grid_resolution[1]);

  // Multiply passed grid values by corresponding entries of this grid
  for (int i = 0; i < grid_size; i++) 
    grid_values[i] *= voxels.grid_values[i];
}



void R2Grid::
Divide(RNScalar value) 
{
  // Just checking
  if (RNIsZero(value)) return;

  // Multiply by recipricol
  Multiply(1.0 / value);
}



void R2Grid::
Divide(const R2Grid& voxels) 
{
  // Resolutions must be the same (for now)
  assert(grid_resolution[0] == voxels.grid_resolution[0]);
  assert(grid_resolution[1] == voxels.grid_resolution[1]);

  // Divide passed grid values by corresponding entries of this grid
  for (int i = 0; i < grid_size; i++) 
    grid_values[i] /= voxels.grid_values[i];
}



void R2Grid::
Threshold(RNScalar threshold, RNScalar low, RNScalar high) 
{
  // Set grid value to low (high) if less/equal (greater) than threshold
  RNScalar *grid_valuep = grid_values;
  for (int i = 0; i < grid_size; i++) {
    if (*grid_valuep <= threshold) *grid_valuep = low;
    else *grid_valuep = high;
    grid_valuep++;
  }
}



void R2Grid::
SquaredDistanceTransform(void)
{
#if 1
  RNAbort("Not implemented");
#else
  int x,y,z,s,t;
  int dist,square,new_dist;
  int* oldBuffer;
  int* newBuffer;
  int first;
  int i;

  // Allocate temporary buffers
  int res = XResolution();
  if (res < YResolution()) res = YResolution();
  if (res < ZResolution()) res = ZResolution();
  oldBuffer = new int[res];
  assert(oldBuffer);
  newBuffer = new int[res];
  assert(newBuffer);

  // Initalize values (0 if was set, max_value if not)
  RNScalar max_value = 3 * (res+1) * (res+1) * (res+1);
  RNScalar *grid_valuesp = grid_values;
  for (i = grid_size; i >= 0; i--) {
    if (*grid_valuesp == 0.0) *grid_valuesp = max_value;
    else *grid_valuesp = 0.0;
    grid_valuesp++;
  }
	
  // Scan along x axis
  for (x = 0; x < XResolution(); x++) {
    for (y = 0; y < YResolution(); y++) {
      first = 1;
      dist = 0;
      for (z = 0; z < ZResolution(); z++) {
        if (GridValue(x,y,z) == 0.0) {
          dist=0;
          first=0;
          SetGridValue(x, y, z, 0);
        }
        else if (first == 0) {
          dist++;
          square = dist*dist;
          SetGridValue(x, y, z, square);
        }
      }
			
      // backward scan
      dist = 0;
      first = 1;
      for (z = ZResolution()-1; z >= 0; z--) {
        if (GridValue(x,y,z) == 0.0){
          dist = 0;
          first = 0;
          SetGridValue(x, y, z, 0);
        }
        else if (first == 0) {
          dist++;
          square = dist*dist;
          if (square < GridValue(x, y, z)) {
            SetGridValue(x, y, z, square);
          }
        }
      }
    }
  } 

  // Scan along y axis
  for (z = 0; z < ZResolution(); z++) {
    for (y = 0; y < YResolution(); y++) {
      // Copy grid values
      for(x = 0; x < XResolution(); x++) 
        oldBuffer[x] = (int) GridValue(x, y, z);
		
      // forward scan
      s = 0;
      for (x = 0; x < XResolution(); x++) {
        dist = oldBuffer[x];
        if (dist) {
          for(t = s; t <= x; t++) {
            new_dist = oldBuffer[t] + (x - t) * (x - t);
            if (new_dist <= dist) {
              dist = new_dist;
              s = t;
            }
          }
        }
        else {
          s = x;
        }
        newBuffer[x] = dist;
      }
			
      // backwards scan
      s = XResolution() - 1;
      for (x = XResolution()-1; x >= 0 ; x--) {
        dist = newBuffer[x];
        if (dist) {
          for (t = s; t >= x; t--) {
            new_dist = oldBuffer[t] + (x - t) * (x - t);
            if (new_dist <= dist) {
              dist = new_dist;
              s = t;
            }
          }
          SetGridValue(x, y, z, dist);
        }
        else {
          s=x;
        }
      }
    }
  }
		
  // along z axis
  for (z = 0; z < ZResolution(); z++) {
    for (x = 0; x < XResolution(); x++) {
      // Copy grid values
      for (y = 0; y < YResolution(); y++)
        oldBuffer[y] = GridValue(x, y, z);
			
      // forward scan
      s = 0;
      for (y = 0; y < YResolution(); y++) {
        dist = oldBuffer[y];
        if (dist) {
          for (t = s; t <= y ; t++) {
            new_dist = oldBuffer[t] + (y - t) * (y - t);
            if (new_dist <= dist){
              dist = new_dist;
              s = t;
            }
          }
        }
        else { 
          s = y;
        }
        newBuffer[y] = dist;
      }

      // backward scan
      s = YResolution() - 1;
      for(y = YResolution()-1; y >=0 ; y--) {
        dist = newBuffer[y];
        if (dist) {
          for (t = s; t > y ; t--) {
            new_dist = oldBuffer[t] + (y - t) * (y - t);
            if (new_dist <= dist){
              dist = new_dist;
              s = t;
            }
          }
          SetGridValue(x, y, z, dist);
        }
        else { 
          s = y; 
        }
      }
    }
  }
	
  // Delete temporary buffers
  delete[] oldBuffer;
  delete[] newBuffer;
#endif
}



void R2Grid::
RasterizeGridPoint(RNScalar x, RNScalar y, RNScalar value)
{
  // Check if within bounds
  if ((x < 0) || (x > grid_resolution[0]-1)) return;
  if ((y < 0) || (y > grid_resolution[1]-1)) return;

  // Bilinear interpolation
  int ix1 = (int) x;
  int iy1 = (int) y;
  int ix2 = ix1 + 1;
  int iy2 = iy1 + 1;
  if (ix2 >= grid_resolution[0]) ix2 = ix1;
  if (iy2 >= grid_resolution[1]) iy2 = iy1;
  float dx = x - ix1;
  float dy = y - iy1;
  (*this)(ix1, iy1) += value * (1.0-dx) * (1.0-dy);
  (*this)(ix1, iy2) += value * (1.0-dx) * dy;
  (*this)(ix2, iy1) += value * dx * (1.0-dy);
  (*this)(ix2, iy2) += value * dx * dy;
}



void R2Grid::
RasterizeGridSpan(const int p1[2], const int p2[2], RNScalar value)
{
  // Get some convenient variables
  int d[2],p[2],dd[2],s[2];
  for (int i = 0; i < 2; i++) {
    d[i]= p2[i] - p1[i];
    if(d[i]<0){
      dd[i] = -d[i];
      s[i] = -1;
    }
    else{
      dd[i] = d[i];
      s[i] = 1;
    }
    p[i] = p1[i];
  }

  // Choose dimensions
  int i1=0;
  if(dd[1]>dd[i1]){i1=1;}
  int i2=(i1+1)%2;

  // Check span extent
  if(dd[i1]==0){
    // Span is a point - rasterize it
    AddGridValue(p[0], p[1], value);
  }
  else {
    // Step along span
    int off[2] = { 0, 0 };
    for (int i = 0; i <= dd[i1]; i++) {
      AddGridValue(p[0], p[1], value);
      off[i2]+=dd[i2];
      p[i1]+=s[i1];
      p[i2]+=s[i2]*off[i2]/dd[i1];
      off[i2]%=dd[i1];
    }
  }
}



void R2Grid::
RasterizeGridTriangle(const int p1[2], const int p2[2], const int p3[2], RNScalar value)
{
  RNAbort("Not implemented");
}



void R2Grid::
RasterizeGridCircle(const R2Point& center, RNLength radius, RNScalar value)
{
  // Figure out the min and max in each dimension
  int mn[2], mx[2];
  for (int i = 0; i < 2; i++) {
    mx[i]= (int) (center[i]+radius);
    if (mx[i] < 0) return;
    if (mx[i] > Resolution(i)-1) mx[i] = Resolution(i)-1;
    mn[i]= (int) (center[i]-radius);
    if (mn[i] > Resolution(i)-1) return;
    if (mn[i] < 0) mn[i] = 0;
  }

  // Rasterize circle interior
  int y1 = (int) (center[1] - radius + 0.5);
  int y2 = (int) (center[1] + radius + 0.5);
  if (y1 < mn[1]) y1 = mn[1];
  if (y2 > mx[1]) y2 = mx[1];
  RNLength radius_squared = radius * radius;
  for (int j = y1; j <= y2; j++) {
    RNCoord y = j - center[1];
    RNCoord y_squared = y*y;
    RNLength x_squared = radius_squared - y_squared;
    RNLength x = sqrt(x_squared);
    int x1 = (int) (center[0] - x + 0.5);
    int x2 = (int) (center[0] + x + 0.5);
    if (x1 < mn[0]) x1 = mn[0];
    if (x2 > mx[0]) x2 = mx[0];
    for (int i = x1; i <= x2; i++) {
      AddGridValue(i, j, value);
    }
  }
}



RNScalar R2Grid::
Dot(const R2Grid& voxels) const
{
  // Resolutions and transforms must be the same (for now)
  assert(grid_resolution[0] == voxels.grid_resolution[0]);
  assert(grid_resolution[1] == voxels.grid_resolution[1]);

  // Compute dot product between this and grid
  RNScalar dot = 0.0;
  for (int i = 0; i < grid_size; i++) 
    dot += grid_values[i] * voxels.grid_values[i];
  return dot;
}



RNScalar R2Grid::
L1Distance(const R2Grid& voxels) const
{
  // Compute distance between this and grid
  RNScalar distance = 0.0;
  for (int i = 0; i < grid_size; i++) 
    distance += fabs(grid_values[i] - voxels.grid_values[i]);
  return distance;
}



RNScalar R2Grid::
L2DistanceSquared(const R2Grid& voxels) const
{
  // Compute distance between this and grid
  RNScalar distance_squared = 0.0;
  for (int i = 0; i < grid_size; i++) {
    RNScalar delta = (grid_values[i] - voxels.grid_values[i]);
    distance_squared += delta * delta;
  }

  // Return result
  return distance_squared;
}



void R2Grid::
SetWorldToGridTransformation(const R2Affine& affine)
{
  // Set transformations
  world_to_grid_transform = affine;
  grid_to_world_transform = affine.Inverse();
  world_to_grid_scale_factor = affine.ScaleFactor();
}



void R2Grid::
SetWorldToGridTransformation(const R2Box& world_box)
{
  // Just checking
  if (grid_size == 0) return;
  if (world_box.NDimensions() < 2) return;

  // Compute grid origin
  R2Vector grid_diagonal(XResolution()-1, YResolution()-1);
  R2Vector grid_origin = 0.5 * grid_diagonal;

  // Compute world origin
  R2Vector world_diagonal(world_box.XLength(), world_box.YLength());
  R2Vector world_origin = world_box.Centroid().Vector();

  // Compute scale
  RNScalar scale = FLT_MAX;
  RNScalar xscale = grid_diagonal[0] / world_diagonal[0];
  if (xscale < scale) scale = xscale;
  RNScalar yscale = grid_diagonal[1] / world_diagonal[1];
  if (yscale < scale) scale = yscale;

  // Compute world-to-grid transformation
  R2Affine affine(R2identity_affine);
  affine.Translate(grid_origin);
  affine.Scale(scale);
  affine.Translate(-world_origin);

  // Set transformations
  SetWorldToGridTransformation(affine);
}



void R2Grid::
SetWorldToGridTransformation(const R2Point& world_origin, const R2Vector& world_xaxis, RNLength world_radius)
{
  // Just checking
  if (grid_size == 0) return;

  // Compute grid origin
  R2Vector grid_diagonal(XResolution()-1, YResolution()-1);
  R2Vector grid_origin = 0.5 * grid_diagonal;
  RNScalar grid_radius = grid_origin[0];
  if (grid_origin[1] < grid_radius) grid_radius = grid_origin[1];

  // Compute scale
  if (RNIsZero(world_radius)) return;
  if (RNIsZero(grid_radius)) return;
  RNScalar scale = grid_radius / world_radius;

  // Compute rotation
  RNAngle rotation = R2InteriorAngle(world_xaxis, R2posx_vector);
  if (world_xaxis.Y() < 0.0) rotation = RN_TWO_PI - rotation;

  // Compute world-to-grid transformation
  R2Affine affine(R2identity_affine);
  affine.Translate(grid_origin);
  affine.Rotate(-rotation);
  affine.Scale(scale);
  affine.Translate(-(world_origin.Vector()));

  // Set transformations
  SetWorldToGridTransformation(affine);
}



R2Point R2Grid::
WorldPosition(RNCoord x, RNCoord y) const
{
  // Transform point from grid coordinates to world coordinates
  R2Point world_point(x, y);
  world_point.Transform(grid_to_world_transform);
  return world_point;

}



R2Point R2Grid::
GridPosition(RNCoord x, RNCoord y) const
{
  // Transform point from world coordinates to grid coordinates
  R2Point grid_point(x, y);
  grid_point.Transform(world_to_grid_transform);
  return grid_point;
}



int R2Grid::
ReadFile(const char *filename)
{
  // Open file
  FILE *fp = fopen(filename, "rb");
  if (!fp) {
    RNFail("Unable to open file %s", filename);
    return 0;
  }

  // Read 
  int status = Read(fp);

  // Close file
  fclose(fp);

  // Return status
  return status;
}



int R2Grid::
WriteFile(const char *filename) const
{
  // Open file
  FILE *fp = fopen(filename, "wb");
  if (!fp) {
    RNFail("Unable to open file %s", filename);
    return 0;
  }

  // Read 
  int status = Write(fp);

  // Close file
  fclose(fp);

  // Return status
  return status;
}



int R2Grid::
Read(FILE *fp)
{
  // Check file
  if (!fp) fp = stdin;

  // Read grid resolution from file
  if (fread(&grid_resolution, sizeof(int), 2, fp) != 2) {
    RNFail("Unable to read resolution from grid file");
    return 0;
  }

  // Update grid resolution variables
  grid_row_size = grid_resolution[0];
  grid_size = grid_row_size * grid_resolution[1];
  if (grid_size <= 0) {
    RNFail("Invalid grid size (%d) in file", grid_size);
    return 0;
  }

  // Read world_to_grid transformation from file
  RNScalar m[9];
  if (fread(m, sizeof(RNScalar), 9, fp) != 9) {
    RNFail("Unable to read transformation matrix from file");
    return 0;
  }

  // Update transformation variables
  world_to_grid_transform.Reset(R3Matrix(m));
  world_to_grid_scale_factor = world_to_grid_transform.ScaleFactor();
  grid_to_world_transform = world_to_grid_transform.Inverse();

  // Allocate grid values
  grid_values = new RNScalar [ grid_size ];
  assert(grid_values);

  // Read grid values
  if (fread(grid_values, sizeof(RNScalar), grid_size, fp) != (unsigned int) grid_size) {
    RNFail("Unable to read %d grid values from file", grid_size);
    return 0;
  }

  // Return number of grid values read
  return grid_size;
}



int R2Grid::
Write(FILE *fp) const
{
  // Check file
  if (!fp) fp = stdout;

  // Write grid resolution from file
  if (fwrite(&grid_resolution, sizeof(int), 2, fp) != 2) {
    RNFail("Unable to write resolution to file");
    return 0;
  }

  // Write world_to_grid transformation to file
  const RNScalar *m = &(world_to_grid_transform.Matrix()[0][0]);
  if (fwrite(m, sizeof(RNScalar), 9, fp) != 9) {
    RNFail("Unable to write transformation matrix to file");
    return 0;
  }

  // Write grid values
  if (fwrite(grid_values, sizeof(RNScalar), grid_size, fp) != (unsigned int) grid_size) {
    RNFail("Unable to write grid values to file");
    return 0;
  }

  // Return number of grid values written
  return grid_size;
}



int R2Grid::
Print(FILE *fp) const
{
  // Check file
  if (!fp) fp = stdout;

  // Print values
  for (int j = 0; j < YResolution(); j++) {
    for (int i = 0; i < XResolution(); i++) {
      fprintf(fp, "%g ", GridValue(i, j));
    }
    fprintf(fp, "\n");
  }

  // Return number of grid values written
  return grid_size;
}




