// Source file for the mesh viewer program



// Include files 

#include "R3Graphics/R3Graphics.h"
#include <GL/glut.h>



// Program variables

static RNArray<char *> mesh_names;
static char *image_name = NULL;
static int print_verbose = 0;



// GLUT variables 

static int GLUTwindow = 0;
static int GLUTwindow_height = 480;
static int GLUTwindow_width = 640;
static int GLUTmouse[2] = { 0, 0 };
static int GLUTbutton[3] = { 0, 0, 0 };
static int GLUTmodifiers = 0;



// Application variables

static R3Viewer *viewer = NULL;
static RNArray<R3Mesh *> meshes;



// Display variables

static int show_faces = 1;
static int show_edges = 0;
static int show_vertices = 0;
static int show_backfacing = 0;
static int show_segments = 0;
static int show_axes = 0;
static int current_mesh = 0;



void GLUTStop(void)
{
  // Destroy window 
  glutDestroyWindow(GLUTwindow);

  // Exit
  exit(0);
}



void GLUTRedraw(void)
{
  // Set viewing transformation
  viewer->Camera().Load();

  // Clear window 
  glClearColor(1.0, 1.0, 1.0, 1.0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Set backface culling
  if (show_backfacing) glDisable(GL_CULL_FACE);
  else glEnable(GL_CULL_FACE);

  // Set lights
  static GLfloat light0_position[] = { 3.0, 4.0, 5.0, 0.0 };
  glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
  static GLfloat light1_position[] = { -3.0, -2.0, -3.0, 0.0 };
  glLightfv(GL_LIGHT1, GL_POSITION, light1_position);

  // Draw every mesh
  for (int i = 0; i < meshes.NEntries(); i++) {
    R3Mesh *mesh = meshes[i];
    if (i != current_mesh) continue;

    // Draw faces
    if (show_faces) {
      glEnable(GL_LIGHTING);
      static GLfloat default_material[] = { 0.8, 0.8, 0.8, 1.0 };
      glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, default_material); 
      if (show_segments) {
        for (int i = 0; i < mesh->NFaces(); i++) {
          R3MeshFace *face = mesh->Face(i);
          int segment = mesh->FaceSegment(face);
          GLfloat r = 31 + 48 * ((3*segment) % 5);
          GLfloat g = 31 + 32 * ((5*segment) % 7);
          GLfloat b = 31 + 19 * ((7*segment) % 11);
          GLfloat material[] = { r / 256.0, g / 256.0, b / 256.0, 1.0 };
          glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, material); 
          mesh->DrawFace(face);
        }
      }
      else {
        mesh->DrawFaces();
      }
    }
   
    // Draw edges
    if (show_edges) {
      glDisable(GL_LIGHTING);
      glColor3f(1.0, 0.0, 0.0);
      mesh->DrawEdges();
    }

    // Draw vertices
    if (show_vertices) {
      glEnable(GL_LIGHTING);
      static GLfloat material[] = { 0.8, 0.4, 0.2, 1.0 };
      glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, material); 
      // mesh->DrawVertices();
      RNScalar radius = 0.005 * mesh->BBox().DiagonalLength();
      for (int i = 0; i < mesh->NVertices(); i++) {
        R3MeshVertex *vertex = mesh->Vertex(i);
        R3Point position = mesh->VertexPosition(vertex);
        position += 0.5 * radius * R3Vector(RNRandomScalar(), RNRandomScalar(), RNRandomScalar());
        material[0] = (31 + 48 * ((3*i) % 5)) / 256.0;
        material[1] = (31 + 32 * ((5*i) % 7)) / 256.0;
        material[2] = (31 + 19 * ((7*i) % 11)) / 256.0;
        glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, material); 
        R3Sphere(position, radius).Draw();
      }
    }
  }

  // Draw axes
  if (show_axes) {
    RNScalar d = meshes[0]->BBox().DiagonalRadius();
    glDisable(GL_LIGHTING);
    glLineWidth(3);
    R3BeginLine();
    glColor3f(1, 0, 0);
    R3LoadPoint(R3zero_point + d * R3negx_vector);
    R3LoadPoint(R3zero_point + d * R3posx_vector);
    R3EndLine();
    R3BeginLine();
    glColor3f(0, 1, 0);
    R3LoadPoint(R3zero_point + d * R3negy_vector);
    R3LoadPoint(R3zero_point + d * R3posy_vector);
    R3EndLine();
    R3BeginLine();
    glColor3f(0, 0, 1);
    R3LoadPoint(R3zero_point + d * R3negz_vector);
    R3LoadPoint(R3zero_point + d * R3posz_vector);
    R3EndLine();
    glLineWidth(1);
  }

  // Capture image and exit
  if (image_name) {
    R2Image image(GLUTwindow_width, GLUTwindow_height, 3);
    image.Capture();
    image.Write(image_name);
    GLUTStop();
  }

  // Swap buffers 
  glutSwapBuffers();
}    



void GLUTResize(int w, int h)
{
  // Resize window
  glViewport(0, 0, w, h);

  // Resize viewer viewport
  viewer->ResizeViewport(0, 0, w, h);

  // Remember window size 
  GLUTwindow_width = w;
  GLUTwindow_height = h;

  // Redraw
  glutPostRedisplay();
}



void GLUTMotion(int x, int y)
{
  // Invert y coordinate
  y = GLUTwindow_height - y;

  // Compute mouse movement
  int dx = x - GLUTmouse[0];
  int dy = y - GLUTmouse[1];
  
  // World in hand navigation 
  R3Point origin = meshes[0]->BBox().Centroid();
  if (GLUTbutton[0]) viewer->RotateWorld(1.0, origin, x, y, dx, dy);
  else if (GLUTbutton[1]) viewer->ScaleWorld(1.0, origin, x, y, dx, dy);
  else if (GLUTbutton[2]) viewer->TranslateWorld(1.0, origin, x, y, dx, dy);
  if (GLUTbutton[0] || GLUTbutton[1] || GLUTbutton[2]) glutPostRedisplay();

  // Remember mouse position 
  GLUTmouse[0] = x;
  GLUTmouse[1] = y;
}



void GLUTMouse(int button, int state, int x, int y)
{
  // Invert y coordinate
  y = GLUTwindow_height - y;
  
  // Process mouse button event

  // Remember button state 
  int b = (button == GLUT_LEFT_BUTTON) ? 0 : ((button == GLUT_MIDDLE_BUTTON) ? 1 : 2);
  GLUTbutton[b] = (state == GLUT_DOWN) ? 1 : 0;

  // Remember modifiers 
  GLUTmodifiers = glutGetModifiers();

   // Remember mouse position 
  GLUTmouse[0] = x;
  GLUTmouse[1] = y;

  // Redraw
  glutPostRedisplay();
}



void GLUTSpecial(int key, int x, int y)
{
  // Invert y coordinate
  y = GLUTwindow_height - y;

  // Process keyboard button event 

  // Remember mouse position 
  GLUTmouse[0] = x;
  GLUTmouse[1] = y;

  // Remember modifiers 
  GLUTmodifiers = glutGetModifiers();

  // Redraw
  glutPostRedisplay();
}



void GLUTKeyboard(unsigned char key, int x, int y)
{
  // Process keyboard button event 
  switch (key) {
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
    if (key - '1' < meshes.NEntries()) current_mesh = key - '1';
    else printf("Unable to select mesh %d\n", key - '1');
    break;

  case 'A':
  case 'a':
    show_axes = !show_axes;
    break;

  case 'B':
  case 'b':
    show_backfacing = !show_backfacing;
    break;

  case 'C': 
  case 'c': {
    // Print camera
    const R3Camera& camera = viewer->Camera();
    printf("#camera  %g %g %g  %g %g %g  %g %g %g  %g \n",
           camera.Origin().X(), camera.Origin().Y(), camera.Origin().Z(),
           camera.Towards().X(), camera.Towards().Y(), camera.Towards().Z(),
           camera.Up().X(), camera.Up().Y(), camera.Up().Z(),
           camera.YFOV());
    }
    break;
      
  case 'E':
  case 'e':
    show_edges = !show_edges;
    break;

  case 'F':
  case 'f':
    show_faces = !show_faces;
    break;

  case 'V':
  case 'v':
    show_vertices = !show_vertices;
    break;

  case 'S':
  case 's':
    show_segments = !show_segments;
    break;

  case 27: // ESCAPE
    GLUTStop();
    break;
  }

  // Remember mouse position 
  GLUTmouse[0] = x;
  GLUTmouse[1] = GLUTwindow_height - y;

  // Remember modifiers 
  GLUTmodifiers = glutGetModifiers();

  // Redraw
  glutPostRedisplay();  
}




#if 0

void GLUTIdle(void)
{
  // Set current window
  if ( glutGetWindow() != GLUTwindow ) 
    glutSetWindow(GLUTwindow);  

  // Redraw
  glutPostRedisplay();
}

#endif



void GLUTInit(int *argc, char **argv)
{
  // Open window 
  glutInit(argc, argv);
  glutInitWindowPosition(100, 100);
  glutInitWindowSize(GLUTwindow_width, GLUTwindow_height);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH); // | GLUT_STENCIL
  GLUTwindow = glutCreateWindow("OpenGL Viewer");

  // Initialize background color 
  glClearColor(200.0/255.0, 200.0/255.0, 200.0/255.0, 1.0);

  // Initialize lights
  static GLfloat lmodel_ambient[] = { 0.2, 0.2, 0.2, 1.0 };
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
  static GLfloat light0_diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light0_diffuse);
  glEnable(GL_LIGHT0);
  static GLfloat light1_diffuse[] = { 0.5, 0.5, 0.5, 1.0 };
  glLightfv(GL_LIGHT1, GL_DIFFUSE, light1_diffuse);
  glEnable(GL_LIGHT1);
  glEnable(GL_NORMALIZE);
  glEnable(GL_LIGHTING);

  // Initialize graphics modes 
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_CULL_FACE);

  // Initialize GLUT callback functions 
  glutDisplayFunc(GLUTRedraw);
  glutReshapeFunc(GLUTResize);
  glutKeyboardFunc(GLUTKeyboard);
  glutSpecialFunc(GLUTSpecial);
  glutMouseFunc(GLUTMouse);
  glutMotionFunc(GLUTMotion);
}



void GLUTMainLoop(void)
{
  // Run main loop -- never returns 
  glutMainLoop();
}



int
ReadMeshes(const RNArray<char *>& mesh_names)
{
  // Read each mesh
  for (int i = 0; i < mesh_names.NEntries(); i++) {
    char *mesh_name = mesh_names[i];

    // Allocate mesh
    R3Mesh *mesh = new R3Mesh();
    assert(mesh);

    // Read mesh from file
    if (!mesh->ReadFile(mesh_name)) {
      delete mesh;
      return 0;
    }

    // Add mesh to list
    meshes.Insert(mesh);
  }

  // Return number of meshes
  return meshes.NEntries();
}



static R3Viewer *
CreateBirdsEyeViewer(R3Mesh *mesh)
{
    // Setup camera view looking down the Z axis
    R3Box bbox = mesh->BBox();
    assert(!bbox.IsEmpty());
    RNLength r = bbox.DiagonalRadius();
    assert((r > 0.0) && RNIsFinite(r));
    static R3Vector towards = R3Vector(-0.57735, -0.57735, -0.57735);
    static R3Vector up = R3Vector(-0.57735, 0.57735, 0.5773);
    R3Point origin = bbox.Centroid() - towards * (2.5 * r);
    R3Camera camera(origin, towards, up, 0.4, 0.4, 0.1 * r, 1000.0 * r);
    R2Viewport viewport(0, 0, GLUTwindow_width, GLUTwindow_height);
    return new R3Viewer(camera, viewport);
}



int ParseArgs(int argc, char **argv)
{
  // Check number of arguments
  if ((argc == 2) && (*argv[1] == '-')) {
    printf("Usage: meshviewer filename [options]\n");
    exit(0);
  }

  // Parse arguments
  argc--; argv++;
  while (argc > 0) {
    if ((*argv)[0] == '-') {
      if (!strcmp(*argv, "-v")) { print_verbose = 1; }
      else if (!strcmp(*argv, "-image")) { argc--; argv++; image_name = *argv; }
      else if (!strcmp(*argv, "-vertices")) { show_vertices = TRUE; }
      else if (!strcmp(*argv, "-edges")) { show_edges = TRUE; }
      else if (!strcmp(*argv, "-back")) { show_backfacing = TRUE; }
      else if (!strcmp(*argv, "-axes")) { show_axes = TRUE; }
      else { fprintf(stderr, "Invalid program argument: %s", *argv); exit(1); }
      argv++; argc--;
    }
    else {
      mesh_names.Insert(*argv);
      argv++; argc--;
    }
  }

  // Check mesh filename
  if (mesh_names.IsEmpty()) {
    fprintf(stderr, "You did not specify a mesh file name.\n");
    return 0;
  }

  // Return OK status 
  return 1;
}



int main(int argc, char **argv)
{
  // Initialize GLUT
  GLUTInit(&argc, argv);

  // Parse program arguments
  if (!ParseArgs(argc, argv)) exit(-1);

  // Read mesh
  if (!ReadMeshes(mesh_names)) exit(-1);

  // Create viewer
  viewer = CreateBirdsEyeViewer(meshes[0]);
  if (!viewer) exit(-1);

  // Run GLUT interface
  GLUTMainLoop();

  // Return success 
  return 0;
}

















