// Program to generate points on surface of mesh



// Include files

#include "R3Shapes/R3Shapes.h"



// Constant definitions

enum {
  SURFACE,
  EDGE,
  VERTEX,
  CENTER_OF_MASS
};

enum {
  AREA,
  CURVATURE
};



// Program arguments

static char *mesh_name  = NULL;
static char *points_name = NULL;
static int point_location = SURFACE;
static int point_weighting = AREA;
static RNBoolean ascii = FALSE;
static RNBoolean print_verbose = FALSE;
static int npoints = 1024;



// Type definitions

struct Point {
  Point(void) : position(0,0,0), normal(0,0,0) {};
  Point(const R3Point& position, const R3Vector& normal) : position(position), normal(normal) {};
  R3Point position;
  R3Vector normal;
};



static R3Mesh *
ReadMesh(char *mesh_name)
{
  // Start statistics
  RNTime start_time;
  start_time.Read();

  // Allocate mesh
  R3Mesh *mesh = new R3Mesh();
  assert(mesh);

  // Read mesh from file
  if (!mesh->ReadFile(mesh_name)) {
    delete mesh;
    return NULL;
  }

  // Print statistics
  if (print_verbose) {
    printf("Read mesh ...\n");
    printf("  Time = %.2f seconds\n", start_time.Elapsed());
    printf("  # Faces = %d\n", mesh->NFaces());
    printf("  # Edges = %d\n", mesh->NEdges());
    printf("  # Vertices = %d\n", mesh->NVertices());
    fflush(stdout);
  }

  // Return success
  return mesh;
}



static RNArray<Point *> *
SelectSurfacePoints(R3Mesh *mesh, int npoints, int point_weighting)
{
  // Allocate array of points
  RNArray<Point *> *points = new RNArray<Point *>();
  if (!points) {
    fprintf(stderr, "Unable to allocate array of points\n");
    return NULL;
  }

  // Count total value of faces
  RNArea total_value = 0.0;
  for (int i = 0; i < mesh->NFaces(); i++) {
    R3MeshFace *face = mesh->Face(i);
    RNScalar face_value = 0;
    if (point_weighting == AREA) face_value = mesh->FaceArea(face);
    else { fprintf(stderr, "Invalid point_weighting: %d\n", point_weighting); return NULL; }
    mesh->SetFaceValue(face, face_value);
    total_value += face_value;
  }
    
  // Generate points
  RNSeedRandomScalar();
  for (int i = 0; i < mesh->NFaces(); i++) {
    R3MeshFace *face = mesh->Face(i);

    // Get vertex positions
    R3MeshVertex *v0 = mesh->VertexOnFace(face, 0);
    R3MeshVertex *v1 = mesh->VertexOnFace(face, 1);
    R3MeshVertex *v2 = mesh->VertexOnFace(face, 2);
    const R3Point& p0 = mesh->VertexPosition(v0);
    const R3Point& p1 = mesh->VertexPosition(v1);
    const R3Point& p2 = mesh->VertexPosition(v2);
    const R3Vector& n0 = mesh->VertexNormal(v0);
    const R3Vector& n1 = mesh->VertexNormal(v1);
    const R3Vector& n2 = mesh->VertexNormal(v2);

    // Determine number of points for face 
    RNScalar ideal_face_npoints = npoints * mesh->FaceValue(face) / total_value;
    int face_npoints = (int) ideal_face_npoints;
    RNScalar remainder = ideal_face_npoints - face_npoints;
    if (remainder > RNRandomScalar()) face_npoints++;

    // Generate random points in face
    for (int j = 0; j < face_npoints; j++) {
      RNScalar r1 = sqrt(RNRandomScalar());
      RNScalar r2 = RNRandomScalar();
      RNScalar t0 = (1.0 - r1);
      RNScalar t1 = r1 * (1.0 - r2);
      RNScalar t2 = r1 * r2;
      R3Point position = t0*p0 + t1*p1 + t2*p2;
      R3Vector normal = t0*n0 + t1*n1 + t2*n2; normal.Normalize();
      Point *point = new Point(position, normal);
      points->Insert(point);
    }
  }

  // Return points
  return points;
}



static RNArray<Point *> *
SelectVertexPoints(R3Mesh *mesh, int npoints, int point_weighting)
{
  // Check number of points
  if (npoints > mesh->NVertices()) npoints = mesh->NVertices();

  // Allocate array of points
  RNArray<Point *> *points = new RNArray<Point *>();
  if (!points) {
    fprintf(stderr, "Unable to allocate array of points\n");
    return NULL;
  }

  // Compute vertex values
  RNScalar total_value = 0;
  for (int i = 0; i < mesh->NFaces(); i++) {
    R3MeshFace *face = mesh->Face(i);
    RNScalar face_value = 0;
    if (point_weighting == AREA) face_value = mesh->FaceArea(face);
    mesh->SetFaceValue(face, face_value);
    for (int j = 0; j < 3; j++) {
      R3MeshVertex *vertex = mesh->VertexOnFace(face, j);
      RNScalar vertex_value = mesh->VertexValue(vertex);
      vertex_value += face_value;
      mesh->SetVertexValue(vertex, vertex_value);
      total_value += face_value;
    }
  }

  // Select vertices weighted by value
  R3mesh_mark++;
  while (npoints--) {
    RNScalar cumulative_value = 0;
    RNScalar target_value = RNRandomScalar() * total_value;
    for (int j = 0; j < mesh->NVertices(); j++) {
      R3MeshVertex *vertex = mesh->Vertex(j);
      if (mesh->VertexMark(vertex) != R3mesh_mark) {
        RNScalar vertex_value = mesh->VertexValue(vertex);
        cumulative_value += vertex_value;
        if ((cumulative_value >= target_value) || (j == mesh->NVertices()-1)) {
          const R3Point& position = mesh->VertexPosition(vertex);
          const R3Vector& normal = mesh->VertexNormal(vertex);
          Point *point = new Point(position, normal);
          points->Insert(point);
          mesh->SetVertexMark(vertex, R3mesh_mark);
          total_value -= vertex_value;
          break;
        }
      }
    }
  }

  // Return points
  return points;
}



static RNArray<Point *> *
SelectCenterOfMass(R3Mesh *mesh)
{
  // Allocate array of points
  RNArray<Point *> *points = new RNArray<Point *>();
  if (!points) {
    fprintf(stderr, "Unable to allocate array of points\n");
    return NULL;
  }

  // Compute mesh center of mass
  RNScalar mesh_area = 0.0;
  R3Point mesh_centroid = R3zero_point;
  for (int i = 0; i < mesh->NFaces(); i++) {
    R3MeshFace *face = mesh->Face(i);
    RNArea area = mesh->FaceArea(face);
    mesh_centroid += area * mesh->FaceCentroid(face);
    mesh_area += area;
  }
  mesh_centroid /= mesh_area;

  // Insert mesh centroid
  Point *point = new Point(mesh_centroid, R3zero_vector);
  points->Insert(point);

  // Return points
  return points;
}



static RNArray<Point *> *
SelectPoints(R3Mesh *mesh, int npoints, int point_location, int point_weighting)
{
  // Start statistics
  RNTime start_time;
  start_time.Read();

  // Select points using requested method 
  RNArray<Point *> *points = NULL;
  if (point_location == VERTEX) 
    points = SelectVertexPoints(mesh, npoints, point_weighting);
  else if (point_location == SURFACE) 
    points= SelectSurfacePoints(mesh, npoints, point_weighting);
  else if (point_location == CENTER_OF_MASS) 
    points= SelectCenterOfMass(mesh);

  // Check points
  if (!points) {
    fprintf(stderr, "Error selecting points\n");
    return 0;
  }

  // Print message
  if (print_verbose) {
    fprintf(stdout, "Generated points ...\n");
    printf("  Time = %.2f seconds\n", start_time.Elapsed());
    printf("  Point location = %d\n", point_location);
    printf("  Point weighting = %d\n", point_weighting);
    printf("  # Points = %d\n", points->NEntries());
    fflush(stdout);
  }

  // Return array of points
  return points;
}



RNBoolean 
WritePoints(const RNArray<Point *>& points, const char *filename, RNBoolean ascii)
{
  // Start statistics
  RNTime start_time;
  start_time.Read();

  if (ascii) {
    // Open file
    FILE *fp = stdout;
    if (filename) {
      fp = fopen(filename, "w");
      if (!fp) {
        fprintf(stderr, "Unable to open output file %s\n", filename);
        return FALSE;
      }
    }

    // Write points
    for (int i = 0; i < points.NEntries(); i++) {
      Point *point = points[i];
      fprintf(fp, "%g %g %g\n", point->position.X(), point->position.Y(), point->position.Z());
    }

    // Close file
    if (filename) fclose(fp);
  }
  else {
    // Open file
    FILE *fp = stdout;
    if (filename) {
      fp = fopen(filename, "wb");
      if (!fp) {
        fprintf(stderr, "Unable to open output file %s\n", filename);
        return FALSE;
      }
    }

    // Write points
    float coordinates[6];
    for (int i = 0; i < points.NEntries(); i++) {
      Point *point = points[i];
      coordinates[0] = point->position.X();
      coordinates[1] = point->position.Y();
      coordinates[2] = point->position.Z();
      coordinates[3] = point->normal.X();
      coordinates[4] = point->normal.Y();
      coordinates[5] = point->normal.Z();
      if (fwrite(coordinates, sizeof(float), 6, fp) != (unsigned int) 6) {
        fprintf(stderr, "Unable to write point to output file %s\n", filename);
        return FALSE;
      }
    }

    // Close file
    if (filename) fclose(fp);
  }

  // Print message
  if (print_verbose) {
    fprintf(stdout, "Wrote points to %s ...\n", (filename) ? filename : "stdout");
    printf("  Time = %.2f seconds\n", start_time.Elapsed());
    fflush(stdout);
  }

  // Return success
  return TRUE;
}



int
ParseArgs (int argc, char **argv)
{
  // Parse arguments
  argc--; argv++;
  while (argc > 0) {
    if ((*argv)[0] == '-') {
      if (!strcmp(*argv, "-npoints")) { argv++; argc--; npoints = atoi(*argv); }
      else if (!strcmp(*argv, "-location")) { argc--; argv++; point_location = atoi(*argv); }
      else if (!strcmp(*argv, "-surface")) { point_location = SURFACE; }
      else if (!strcmp(*argv, "-edge")) { point_location = EDGE; }
      else if (!strcmp(*argv, "-vertex")) { point_location = VERTEX; }
      else if (!strcmp(*argv, "-center")) { point_location = CENTER_OF_MASS; }
      else if (!strcmp(*argv, "-weighting")) { argc--; argv++; point_weighting = atoi(*argv); }
      else if (!strcmp(*argv, "-area")) { point_weighting = AREA; }
      else if (!strcmp(*argv, "-ascii")) { ascii = TRUE; }
      else if (!strcmp(*argv, "-v")) { print_verbose = TRUE; }
      else RNAbort("Invalid program argument: %s", *argv);
      argv++; argc--;
    }
    else {
      if (!mesh_name) mesh_name = *argv;
      else if (!points_name) points_name = *argv;
      else RNAbort("Invalid program argument: %s", *argv);
      argv++; argc--;
    }
  }

  // Check mesh name
  if (!mesh_name) {
    // Print usage statement
    fprintf(stderr, 
      "Usage: off2pts <options> input_mesh output_points\n\n"
      "options:\n"
      " -npoints #   (default: 1024)\n"      
      "\n");
    return FALSE;
  }

  // Return success
  return TRUE;
}



int
main(int argc, char **argv)
{
  // Start statistics
  RNTime start_time;
  start_time.Read();

  // Parse args
  if(!ParseArgs(argc, argv)) exit(-1);

  // Read the mesh
  R3Mesh *mesh = ReadMesh(mesh_name);
  if (!mesh) exit(-1);

  // Select points
  RNArray<Point *> *points = SelectPoints(mesh, npoints, point_location, point_weighting);
  if (!points) exit(-1);

  // Write points
  if (!WritePoints(*points, points_name, ascii)) exit(-1);

  // Print message
  if (print_verbose) {
    fprintf(stdout, "Finished in %6.3f seconds.\n", start_time.Elapsed());
    fflush(stdout);
  }

  // Return success
  return 0;
}








