/* By John O'Neill, during Princeton SPE 2010
 * 
 * Creates photographic mosaics.
 * 
 * Note: comments are spares. Email me at
 * jconeill@princeton.edu if you have any
 * questions.
 * 
 * Enjoy!
 * 
 * * * * * * * * * * * * * * * * * * * * * * */

import javax.swing.*;  
import java.awt.event.ActionEvent;
import javax.swing.event.ChangeEvent;
import java.awt.event.ActionListener;
import javax.swing.event.ChangeListener;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import java.awt.FlowLayout;
import java.awt.BorderLayout;
import java.awt.CardLayout;
import javax.swing.BorderFactory;
import java.awt.Color;
import java.awt.Insets;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.io.File;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Dimension;
import java.awt.Toolkit;
import javax.swing.JFileChooser;
import javax.swing.border.TitledBorder;
import javax.swing.border.Border;
import java.awt.Font;
import javax.imageio.ImageIO;
import java.io.IOException;
import java.awt.image.WritableRaster;
import java.awt.image.DirectColorModel;
import java.awt.FileDialog;
import javax.swing.JTabbedPane;
import javax.swing.JProgressBar;
import java.awt.Cursor;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;
import java.awt.event.MouseListener;
import javax.swing.JRadioButton;
import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;

public class MGUI extends JPanel
    implements ActionListener {
    
    private static JFrame frame;   // main frame
    private Picture original;      // uploaded photo with original size
    private BufferedImage scaled;
    private BufferedImage mosaic;
    private BufferedImage mixed; // part scaled, part mosaic (result of ghosting)
    private JPanel jp_options_basic;
    private JLabel jl_rows;
    private JLabel jl_cols;
    private JButton bu_loadmain;
    private JButton bu_loaddb;
    private JButton bu_transform;
    private JButton bu_save;
    private JButton bu_savehr;     // high res
    private JButton bu_magic;
    private JLabel la_main;
    private JLabel jl_hallelujah;
    private JSlider js_row;
    private JSlider js_col;
    private JSlider js_ghost;
    private JSlider js_tolerance;
    private JTextField tf_length;
    private JTextField tf_rows;
    private JTextField tf_cols;
    private JTextField tf_compwidth;
    private JProgressBar pb_hires;
    private JPanel jp_magicbig;
    
    Preview preview;
    
    // jpanel that contains the main image and 
    // uses an override graphics method
    private MainPhoto mainphoto;
    
    private double tolerance = 15.0;
    
    // semi-arbitrary values that causes good window sizes.
    // because frame cannot be resized, its dimension depends on these values.
    // (photos usually follow scalefactor 4:3, meaning that
    // if 720 is the longside, 540 is the shortside, thus:)
    private int LONGSIDE = 720;
    private int SHORTSIDE = 540;
    
    private double scalefactor;
    
    private boolean transformswitch = false;
    private boolean dbloaded = false;
    private boolean showmosaic = false;
    
    private int tilewidth;
    private int tileheight;
    
    // for debugging purposes, this is set
    // otherwise, leave as an empty string
    private String lastdir = "";
    
    private Picture[][] bestmatches;
    
    private boolean guides = false;
    private Color guidecolor = null;
    
    private int previewmin = 90;
    private int previewmax = 200;
    private int currentrow, currentcol;
    
    /*********************************\
      Various components and the like.
      Also includes Component Listener.
      \********************************/
    public MGUI() {
        
        this.setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));
        
        mainphoto = new MainPhoto();
        JPanel jp_mainphoto = new JPanel();
        jp_mainphoto.setLayout(new BoxLayout(jp_mainphoto, BoxLayout.PAGE_AXIS));
        jp_mainphoto.add(Box.createVerticalGlue());
        jp_mainphoto.add(mainphoto);
        jp_mainphoto.add(Box.createVerticalGlue());
        
        // button for choosing main photo
        ImageIcon loadingMainIcon = createImageIcon("loadimg.png");
        bu_loadmain = new JButton("", loadingMainIcon);
        bu_loadmain.setMnemonic(KeyEvent.VK_O);
        bu_loadmain.setActionCommand("loadimg");
        
        // button for choosing folder of tiles
        //ImageIcon dbIconEmpty = createImageIcon("db_empty.png");
        ImageIcon bu_createdbicon = createImageIcon("createdb.png");
        JButton bu_createdb = new JButton("", bu_createdbicon);
        bu_createdb.setMnemonic(KeyEvent.VK_C);
        bu_createdb.setActionCommand("createdb");
        
        // button for choosing folder of tiles
        ImageIcon bu_loadbicon = createImageIcon("loaddb.png");
        bu_loaddb = new JButton("", bu_loadbicon);
        bu_loaddb.setMnemonic(KeyEvent.VK_D);
        bu_loaddb.setActionCommand("loaddb");
        
        // button for creating mosaic
        ImageIcon bu_transformIcon = createImageIcon("transform.png");
        bu_transform = new JButton(bu_transformIcon);
        bu_transform.setMnemonic(KeyEvent.VK_T);
        bu_transform.setActionCommand("transform");
        bu_transform.setEnabled(false);
        
        //ImageIcon bu_saveIcon = createImageIcon("bu_createICON.png");
        //bu_save = new JButton(bu_saveIcon);
        ImageIcon bu_saveicon = createImageIcon("save.png");
        bu_save = new JButton("", bu_saveicon);
        bu_save.setMnemonic(KeyEvent.VK_S);
        bu_save.setActionCommand("save");
        bu_save.setEnabled(false);
        
        // listen for actions on buttons
        bu_loadmain.addActionListener(this);
        bu_loadmain.setToolTipText("Load It!");
        bu_createdb.addActionListener(this);
        bu_createdb.setToolTipText("Create It!");
        bu_loaddb.addActionListener(this);
        bu_loaddb.setToolTipText("Load It!");
        bu_transform.addActionListener(this);
        bu_transform.setToolTipText("Transform It!");
        bu_save.addActionListener(this);
        bu_save.setToolTipText("Save It!");
        
        // panel for main image button
        JPanel jp_main = new JPanel();
        jp_main.setLayout(new BoxLayout(jp_main, BoxLayout.LINE_AXIS));
        jp_main.add(Box.createHorizontalGlue());
        jp_main.add(bu_loadmain);
        jp_main.add(Box.createHorizontalGlue());
        
        final int DIM_SLIDER_MIN = 10;
        final int DIM_SLIDER_MAX = 130;
        final int DIM_SLIDER_INIT = 70;
        js_row = new JSlider(JSlider.HORIZONTAL, DIM_SLIDER_MIN, DIM_SLIDER_MAX, DIM_SLIDER_INIT);
        js_col = new JSlider(JSlider.HORIZONTAL, DIM_SLIDER_MIN, DIM_SLIDER_MAX, DIM_SLIDER_INIT);
        js_row.setMajorTickSpacing(40);
        js_row.setMinorTickSpacing(10);
        js_row.setForeground(Color.WHITE);
        js_row.setPaintTicks(true);
        js_row.setPaintLabels(false);
        js_row.addChangeListener(new SliderListener());
        js_col.setMajorTickSpacing(40);
        js_col.setMinorTickSpacing(10);
        js_col.setForeground(Color.WHITE);
        js_col.setPaintTicks(true);
        js_col.setPaintLabels(false);
        js_col.addChangeListener(new SliderListener());
        
        jl_rows = new JLabel("Rows: " + DIM_SLIDER_INIT, JLabel.CENTER);
        jl_rows.setForeground(Color.WHITE);
        jl_cols = new JLabel("Columns: " + DIM_SLIDER_INIT, JLabel.CENTER);
        jl_cols.setForeground(Color.WHITE);
        
        JPanel jp_rowlabel = new JPanel();
        jp_rowlabel.setLayout(new BoxLayout(jp_rowlabel, BoxLayout.LINE_AXIS));
        jp_rowlabel.add(Box.createHorizontalGlue());
        jp_rowlabel.add(jl_rows);
        jp_rowlabel.add(Box.createHorizontalGlue());
        
        JPanel jp_rowslider = new JPanel();
        jp_rowslider.setLayout(new BoxLayout(jp_rowslider, BoxLayout.LINE_AXIS));
        //jp_rowslider.add(Box.createHorizontalGlue());
        jp_rowslider.add(js_row);
        //jp_rowslider.add(Box.createHorizontalGlue());
        
        JPanel jp_collabel = new JPanel();
        jp_collabel.setLayout(new BoxLayout(jp_collabel, BoxLayout.LINE_AXIS));
        jp_collabel.add(Box.createHorizontalGlue());
        jp_collabel.add(jl_cols);
        jp_collabel.add(Box.createHorizontalGlue());
        
        JPanel jp_colslider = new JPanel();
        jp_colslider.setLayout(new BoxLayout(jp_colslider, BoxLayout.LINE_AXIS));
        //jp_colslider.add(Box.createHorizontalGlue());
        jp_colslider.add(js_col);
        //jp_colslider.add(Box.createHorizontalGlue());
        
        JLabel jl_tiledb = new JLabel("::::::::::::::::::::: Tile Database :::::::::::::::::::::");
        jl_tiledb.setForeground(Color.WHITE);
        JPanel jp_dblabel = new JPanel();
        jp_dblabel.setLayout(new BoxLayout(jp_dblabel, BoxLayout.LINE_AXIS));
        jp_dblabel.add(Box.createHorizontalGlue());
        jp_dblabel.add(jl_tiledb);
        jp_dblabel.add(Box.createHorizontalGlue());
        
        // panel for choosing database
        JPanel jp_db = new JPanel();
        jp_db.setLayout(new BoxLayout(jp_db, BoxLayout.LINE_AXIS));
        jp_db.add(Box.createHorizontalGlue());
        jp_db.add(bu_createdb);
        jp_db.add(bu_loaddb);
        jp_db.add(Box.createHorizontalGlue());
        
        JLabel jl_mosaic = new JLabel(":::::::::::::::::::::::::: Mosiac ::::::::::::::::::::::::::");
        jl_mosaic.setForeground(Color.WHITE);
        JPanel jp_mosaic = new JPanel();
        jp_mosaic.setLayout(new BoxLayout(jp_mosaic, BoxLayout.LINE_AXIS));
        jp_mosaic.add(Box.createHorizontalGlue());
        jp_mosaic.add(jl_mosaic);
        jp_mosaic.add(Box.createHorizontalGlue());
        
        // panel for transformation button
        JPanel jp_transform = new JPanel();
        jp_transform.setLayout(new BoxLayout(jp_transform, BoxLayout.LINE_AXIS));
        jp_transform.add(Box.createHorizontalGlue());
        jp_transform.add(bu_transform);
        jp_transform.add(bu_save);
        jp_transform.add(Box.createHorizontalGlue());
        
        // right side pane containing mosaic options
        jp_options_basic = new JPanel();
        jp_options_basic.setLayout(new BoxLayout(jp_options_basic, BoxLayout.PAGE_AXIS));
        jp_options_basic.add(Box.createVerticalGlue());
        jp_options_basic.add(jp_main);
        jp_options_basic.add(jp_rowlabel);
        jp_options_basic.add(jp_rowslider);
        jp_options_basic.add(jp_collabel);
        jp_options_basic.add(jp_colslider);
        jp_options_basic.add(jp_dblabel);
        jp_options_basic.add(jp_db);
        jp_options_basic.add(jp_mosaic);
        jp_options_basic.add(jp_transform);
        jp_options_basic.add(Box.createVerticalGlue());
        
        JPanel jp_options_adv = new JPanel();
        jp_options_adv.setLayout(new BoxLayout(jp_options_adv, BoxLayout.PAGE_AXIS));
        JPanel jp_compwidth = new JPanel();
        jp_compwidth.setLayout(new BoxLayout(jp_compwidth, BoxLayout.LINE_AXIS));
        JPanel jp_highreslabel = new JPanel();
        jp_highreslabel.setLayout(new BoxLayout(jp_highreslabel, BoxLayout.LINE_AXIS));
        JPanel jp_highreslength = new JPanel();
        JPanel jp_dimensions = new JPanel();
        
        JLabel jl_tweaks = new JLabel(":::::::::::::::::::: Image Tweaks :::::::::::::::::::::");
        jl_tweaks.setForeground(Color.WHITE);
        
        JPanel jp_tweaks = new JPanel();
        jp_tweaks.setLayout(new BoxLayout(jp_tweaks, BoxLayout.LINE_AXIS));
        jp_tweaks.add(Box.createHorizontalGlue());
        jp_tweaks.add(jl_tweaks);
        jp_tweaks.add(Box.createHorizontalGlue());
        
        JLabel jl_compwidth = new JLabel("Monitor width (inches):");
        jl_compwidth.setForeground(Color.WHITE);
        tf_compwidth = new JTextField(5);
        tf_compwidth.setMaximumSize(new Dimension(40, 30));
        jp_compwidth.add(jl_compwidth);
        jp_compwidth.add(Box.createHorizontalGlue());
        jp_compwidth.add(tf_compwidth);
        
        tf_length = new JTextField(5);
        tf_length.setMaximumSize(new Dimension(40, 30));
        
        JLabel jl_highres = new JLabel(":::::::::::: Higher Resolution Mosaic ::::::::::::");
        jl_highres.setForeground(Color.WHITE);
        jp_highreslabel.add(Box.createHorizontalGlue());
        jp_highreslabel.add(jl_highres);
        jp_highreslabel.add(Box.createHorizontalGlue());
        
        jp_highreslength.setLayout(new BoxLayout(jp_highreslength, BoxLayout.LINE_AXIS));
        JLabel jl_length = new JLabel("Length (inches): ");
        jl_length.setForeground(Color.WHITE);
        
        //jp_highreslength.add(Box.createHorizontalGlue());
        jp_highreslength.add(jl_length);
        jp_highreslength.add(Box.createHorizontalGlue());
        jp_highreslength.add(tf_length);
        
        jp_dimensions.setLayout(new BoxLayout(jp_dimensions, BoxLayout.LINE_AXIS));
        JLabel jl_dim = new JLabel("Rows x Columns:");
        jl_dim.setForeground(Color.WHITE);
        tf_rows = new JTextField(5);
        tf_rows.setMaximumSize(new Dimension(60, 30));
        tf_cols = new JTextField(5);
        tf_cols.setMaximumSize(new Dimension(60, 30));
        jp_dimensions.add(jl_dim);
        jp_dimensions.add(Box.createHorizontalGlue());
        jp_dimensions.add(tf_rows);
        jp_dimensions.add(tf_cols);
        
        JPanel jp_savehr = new JPanel();
        jp_savehr.setLayout(new BoxLayout(jp_savehr, BoxLayout.LINE_AXIS));
        bu_savehr = new JButton("Save");
        bu_savehr.setMnemonic(KeyEvent.VK_H);
        bu_savehr.addActionListener(this);
        bu_savehr.setActionCommand("savehr");
        bu_savehr.setEnabled(false);
        pb_hires = new JProgressBar(0, 100);
        pb_hires.setValue(0);
        pb_hires.setStringPainted(true);
        jp_savehr.add(bu_savehr);
        jp_savehr.add(pb_hires);
        //jp_savehr.add(Box.createHorizontalGlue());
        
        JPanel jp_hallelujah = new JPanel();
        jp_hallelujah.setLayout(new BoxLayout(jp_hallelujah, BoxLayout.LINE_AXIS));
        jl_hallelujah = new JLabel("Hallelujah! Your mosaic has been saved!");
        jl_hallelujah.setForeground(Color.WHITE);
        jl_hallelujah.setVisible(false);
        jp_hallelujah.add(Box.createHorizontalGlue());
        jp_hallelujah.add(jl_hallelujah);
        jp_hallelujah.add(Box.createHorizontalGlue());
        
        JPanel jp_ghost = new JPanel();
        jp_ghost.setLayout(new BoxLayout(jp_ghost, BoxLayout.LINE_AXIS));
        JLabel jl_ghost = new JLabel("Source Image Ghosting");
        jl_ghost.setForeground(Color.WHITE);
        jp_ghost.add(jl_ghost);
        jp_ghost.add(Box.createHorizontalGlue());
        
        final int G_SLIDER_MIN = 0; // ghosting slider
        final int G_SLIDER_MAX = 100;
        final int G_SLIDER_INIT = 0;
        js_ghost = new JSlider(JSlider.HORIZONTAL, G_SLIDER_MIN, G_SLIDER_MAX, G_SLIDER_INIT);
        js_ghost.setMajorTickSpacing(20);
        js_ghost.setMinorTickSpacing(5);
        js_ghost.setForeground(Color.WHITE);
        js_ghost.setPaintTicks(false);
        js_ghost.setPaintLabels(false);
        js_ghost.addChangeListener(new SliderListener());
        
        JPanel jp_tolerance = new JPanel();
        jp_tolerance.setLayout(new BoxLayout(jp_tolerance, BoxLayout.LINE_AXIS));
        JLabel jl_tolerance = new JLabel("Matching Tolerance");
        jl_tolerance.setForeground(Color.WHITE);
        jp_tolerance.add(jl_tolerance);
        jp_tolerance.add(Box.createHorizontalGlue());
        
        final int T_SLIDER_MIN = 0;   // tolerance slider
        final int T_SLIDER_MAX = 225;
        final int T_SLIDER_INIT = 15;
        js_tolerance = new JSlider(JSlider.HORIZONTAL, T_SLIDER_MIN, T_SLIDER_MAX, (int)tolerance);
        js_tolerance.setMajorTickSpacing(25);
        js_tolerance.setMinorTickSpacing(5);
        js_tolerance.setForeground(Color.WHITE);
        js_tolerance.setPaintTicks(true);
        js_tolerance.setPaintLabels(false);
        js_tolerance.addChangeListener(new SliderListener());
        
        JLabel jl_guides = new JLabel("Guides: ");
        jl_guides.setForeground(Color.WHITE);
        
        JRadioButton nonebutton = new JRadioButton("None");
        nonebutton.setActionCommand("none");
        nonebutton.setForeground(Color.WHITE);
        nonebutton.setSelected(true);
        nonebutton.addActionListener(this);
        
        JRadioButton blackbutton = new JRadioButton("Black");
        blackbutton.setActionCommand("black");
        blackbutton.setForeground(Color.WHITE);
        blackbutton.addActionListener(this);
        
        JRadioButton whitebutton = new JRadioButton("White");
        whitebutton.setActionCommand("white");
        whitebutton.setForeground(Color.WHITE);
        whitebutton.addActionListener(this);
        
        ButtonGroup group = new ButtonGroup();
        group.add(nonebutton);
        group.add(blackbutton);
        group.add(whitebutton);
        
        JPanel jp_guide = new JPanel();
        jp_guide.setLayout(new BoxLayout(jp_guide, BoxLayout.LINE_AXIS));
        jp_guide.add(jl_guides);
        jp_guide.add(Box.createHorizontalGlue());
        jp_guide.add(nonebutton);
        jp_guide.add(Box.createHorizontalGlue());
        jp_guide.add(blackbutton);
        jp_guide.add(Box.createHorizontalGlue());
        jp_guide.add(whitebutton);
        jp_guide.add(Box.createHorizontalGlue());
        
        ImageIcon bu_magicicon = createImageIcon("magicbox.png");
        bu_magic = new JButton("", bu_magicicon);
        bu_magic.setMnemonic(KeyEvent.VK_M);
        bu_magic.setActionCommand("magic");
        bu_magic.addActionListener(this);
        
        JPanel jp_magicsmall = new JPanel();
        jp_magicsmall.setLayout(new BoxLayout(jp_magicsmall, BoxLayout.LINE_AXIS));
        jp_magicsmall.add(Box.createHorizontalGlue());
        jp_magicsmall.add(bu_magic);
        jp_magicsmall.add(Box.createHorizontalGlue());
        
        jp_magicbig = new JPanel();
        jp_magicbig.setLayout(new BoxLayout(jp_magicbig, BoxLayout.PAGE_AXIS));
        jp_magicbig.add(Box.createVerticalGlue());
        jp_magicbig.add(jp_magicsmall);
    
        jp_options_adv.add(jp_tweaks);
        jp_options_adv.add(Box.createVerticalStrut(7));
        jp_options_adv.add(jp_guide);
        jp_options_adv.add(jp_tolerance);
        jp_options_adv.add(js_tolerance);
        jp_options_adv.add(jp_ghost);
        jp_options_adv.add(js_ghost);
        jp_options_adv.add(Box.createVerticalStrut(13));
        jp_options_adv.add(jp_highreslabel);
        jp_options_adv.add(Box.createVerticalStrut(9));
        jp_options_adv.add(jp_compwidth);
        jp_options_adv.add(jp_highreslength);
        jp_options_adv.add(jp_dimensions);
        jp_options_adv.add(jp_savehr);
        jp_options_adv.add(jp_hallelujah);
        //jp_options_adv.add(jp_magicbig);
        
        JTabbedPane tpain = new JTabbedPane(); // JTabbedPane, for when you're on the big, blue, watery road
        tpain.addTab("Basic Options", jp_options_basic);
        tpain.addTab("Advanced Options", jp_options_adv);
        
        JPanel jp_rightpane = new JPanel();
        jp_rightpane.setLayout(new BoxLayout(jp_rightpane, BoxLayout.PAGE_AXIS));
        jp_rightpane.add(tpain);
        
        add(Box.createHorizontalGlue());
        add(jp_mainphoto);
        add(Box.createHorizontalGlue());
        add(jp_rightpane);
        add(Box.createHorizontalGlue());
        
    }
    
    /*********************************************\
      Create the GUI and show it.  For thread safety,
      this method should be invoked from the
      event-dispatching thread. 
      \*********************************************/
    private static void createAndShowGUI(){
        
        frame = new JFrame("Lapidary");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        //Create and set up the content pane.
        MGUI contentpane = new MGUI();
        contentpane.setOpaque(true); //content panes must be opaque
        
        frame.setBackground(Color.DARK_GRAY);
        frame.setIconImage(new ImageIcon("128.png").getImage());
        frame.setContentPane(contentpane);
        
        //Display the window.
        frame.pack();
        centerWindow();
        frame.setResizable(false);
        frame.requestFocusInWindow();
        frame.setVisible(true); 
    }
    
    /********************\
      ActionEvent listener. 
      \********************/
    public void actionPerformed(ActionEvent e) {
        
        if ("loadimg".equals(e.getActionCommand()))
            setOriginalAndScaled();
        
        if ("loaddb".equals(e.getActionCommand()))
            loadDatabase();
        
        if ("transform".equals(e.getActionCommand()))
            transform();
        
        if ("save".equals(e.getActionCommand()))
            save();
        
        if ("createdb".equals(e.getActionCommand()))
            createDatabase();
        
        if ("savehr".equals(e.getActionCommand()))
            saveHiRes();
        
        if ("black".equals(e.getActionCommand())) {
            guides = true;
            guidecolor = Color.BLACK;
            if (mainphoto != null)
                mainphoto.repaint();
        }
        
        if ("white".equals(e.getActionCommand())) {
            guides = true;
            guidecolor = Color.WHITE;
            if (mainphoto != null)
                mainphoto.repaint();
        }
        
        if ("none".equals(e.getActionCommand())) {
            guides = false;
            guidecolor = null;
            if (mainphoto != null)
                mainphoto.repaint();
        }
        
        if ("magic".equals(e.getActionCommand()))  {
            jp_magicbig.setVisible(false);
        }
        
    }
        
    private class SliderListener implements ChangeListener {
        public void stateChanged(ChangeEvent e) {
            
            JSlider source = (JSlider)e.getSource();
            
            if (source == js_row && source.getValueIsAdjusting()) {
                jl_rows.setText("Rows: " + (int)js_row.getValue());
                mainphoto.repaint();
                frame.pack();
            }
            
            if (source == js_col && source.getValueIsAdjusting()) {
                jl_cols.setText("Cols: " + (int)js_col.getValue());
                mainphoto.repaint();
                frame.pack();
            }
            
            if (source == js_ghost && source.getValueIsAdjusting()) {
                if (mosaic != null && showmosaic == true) {
                    mixed = blender(mosaic, scaled);
                    mainphoto.repaint();
                }
            }
            
            if (source == js_tolerance && source.getValueIsAdjusting())
                tolerance = js_tolerance.getValue();   
            
        }
    }
    
    /****************************************************\
      Returns an ImageIcon, or null if the path is invalid.
      \****************************************************/
    /** Returns an ImageIcon, or null if the path was invalid. */
    protected static ImageIcon createImageIcon(String path) {
        java.net.URL imgURL = MGUI.class.getResource(path);
        if (imgURL != null) {
            return new ImageIcon(imgURL);
        } else {
            System.err.println("Couldn't find file: " + path);
            return null;
        }
    }
    protected static ImageIcon createImageIcon(BufferedImage img) {
        if (img != null) {
            return new ImageIcon(img);
        } else {
            System.err.println("Couldn't use image.");
            return null;
        }
    }
    
    private void setOriginalAndScaled() {
        
        JFileChooser fc = new JFileChooser();
        fc.addChoosableFileFilter(new ImageFilter()); // only search for images
        fc.setAccessory(new ImagePreview(fc));
        int returnVal = fc.showOpenDialog(this);
        
        // if user selects file before closing chooser window
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            
            File file = fc.getSelectedFile();
            original = new Picture(file);      // preserves original photo
            
            makeScaledFromOrig();
            
            reshapeWindow();
            unlockTransformIfReady();
            frame.pack();
            
            transformswitch = false;
            showmosaic = false;
            mosaic = null;
            bu_save.setEnabled(false);
            resetGhost();
            bestmatches = null;
            
        }
        
    }
    
    private void createDatabase(){
        
        JFileChooser folderfinder = new JFileChooser();
        folderfinder.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        
        int returnVal = folderfinder.showOpenDialog(this);
        
        // if user selects folder before closing chooser window
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            
            File folder = folderfinder.getSelectedFile();
            String folderpath = folderfinder.getCurrentDirectory() + File.separator + folder.getName();
            
            FileDialog saver = new FileDialog(frame, "Use a .txt extension to save database", FileDialog.SAVE);
            saver.setVisible(true);
            
            String dbname = saver.getFile();
            String dbpath = saver.getDirectory() + File.separator + dbname;
            if (dbname != null) {
                
                String suffix = dbname.substring(dbname.lastIndexOf('.') + 1);
                
                ArrayList<String> invalid;
                if (suffix.toLowerCase().equals("txt"))
                    invalid = Storer.saveFolder(folderpath, dbpath);
                else {     //if they forget a .txt extension
                    dbpath += ".txt";
                    invalid = Storer.saveFolder(folderpath, dbpath);
                }
                
                if (invalid == null) {
                    ImageIcon icon = createImageIcon("good.png");
                    JOptionPane.showMessageDialog(frame,"Finished compiling database",
                                                  "Databasing complete",
                                                  JOptionPane.INFORMATION_MESSAGE, icon);
                }
                else if (!invalid.isEmpty()) {
                    String msg = "The follow files are not accepted image formats and have not been included:";
                    for (int i = 0; i < invalid.size(); i++) 
                        msg += "\n" + invalid.get(i);
                    
                    ImageIcon icon = createImageIcon("bad.png");
                    JOptionPane.showMessageDialog(frame,msg,
                                                  "Invalid filetypes",
                                                  JOptionPane.WARNING_MESSAGE, icon);
                }
                
            }
            
        }
        
    }
    
    private void loadDatabase() {
        
        JFileChooser fc = new JFileChooser();
        fc.addChoosableFileFilter(new TextFilter()); // only search for images
        
        int returnVal = fc.showOpenDialog(this);
        
        // if user selects file before closing chooser window
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            
            File file = fc.getSelectedFile();
            
            DatabaseNew.load(fc.getCurrentDirectory() + File.separator + file.getName());
            dbloaded = true;
            unlockTransformIfReady();
            frame.pack();
            
        }
        
    }
    
    /*******************\
      The Bread-n'-Butter.
      \*******************/
    private void transform() {
        
        int rows = js_row.getValue();
        int cols = js_col.getValue();
        
        makeScaledFromOrig();
        
        mosaic = new BufferedImage(scaled.getWidth(), scaled.getHeight(), BufferedImage.TYPE_INT_ARGB);
        
        if (frame.getHeight() > mosaic.getHeight())
            reshapeWindow(); // shrink frame to fit
        frame.pack();
        
        tileheight = (int)((double)mosaic.getHeight() / (double)rows);
        tilewidth = (int)((double)mosaic.getWidth() / (double)cols);
        // rounding errors (dealing with ints and divisions)
        while (tileheight*rows < mosaic.getHeight())
            rows++;
        
        while (tilewidth*cols < mosaic.getWidth())
            cols++;
        
        bestmatches = new Picture[rows][cols];
        
        showmosaic = true;
        transformswitch = true;
        
        for (int row = 0; row < rows; row++) {
            for (int col = 0; col < cols; col++) {
                
                // these get adjusted if need by and preserve the variables
                // "tilewidth" and "tileheight"
                int tempwidth = tilewidth;
                int tempheight = tileheight;
                
                // if the next tile is an edge "sliver" and needs adjusting
                if ((row*tileheight + tileheight) > scaled.getHeight())
                    tempheight = scaled.getHeight() - row*tileheight;
                
                if ((col*tilewidth + tilewidth) > scaled.getWidth())
                    tempwidth = scaled.getWidth() - col*tilewidth;
                
                BufferedImage subset = scaled.getSubimage(col*tilewidth, row*tileheight, tempwidth, tempheight);
                Picture bestmatch = Comparer.getBestMatches(subset, tolerance);
                bestmatch.resize(tilewidth, tileheight);
                bestmatches[row][col] = bestmatch;
                this.paintImmediately(this.getX(), this.getY(), this.getWidth(), this.getHeight());
                
            }
            
        }
        //mosaic = mosaic.getSubimage(0, 0, mainphoto.getPreferredSize().width, mainphoto.getPreferredSize().height);
        transformswitch = false;
        
        bu_save.setEnabled(true);
        resetGhost();
        
    }
    
    private void save() {
        
        FileDialog chooser = new FileDialog(frame, "Use a .png, .jpg, or .gif extension", FileDialog.SAVE);
        chooser.setVisible(true);
        String filename = chooser.getFile();
        
        if (filename != null) {
            if (!filename.trim().equals("")) {
                
                String fullpath = chooser.getDirectory() + File.separator + chooser.getFile();
                
                File file = new File(fullpath);
                String suffix = fullpath.substring(fullpath.lastIndexOf('.') + 1);
                
                // png files
                if (suffix.toLowerCase().equals("png")) {
                    try { ImageIO.write(mixed, suffix, file); }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                // need to change from ARGB to RGB for jpeg
                // reference: http://archives.java.sun.com/cgi-bin/wa?A2=ind0404&L=java2d-interest&D=0&P=2727
                else if (suffix.toLowerCase().equals("jpg") || suffix.toLowerCase().equals("jpeg")) {
                    WritableRaster raster = mixed.getRaster();
                    WritableRaster newRaster;
                    newRaster = raster.createWritableChild(0, 0, mixed.getWidth(), mixed.getHeight(), 0, 0, new int[] {0, 1, 2});
                    DirectColorModel cm = (DirectColorModel) mixed.getColorModel();
                    DirectColorModel newCM = new DirectColorModel(cm.getPixelSize(),
                                                                  cm.getRedMask(),
                                                                  cm.getGreenMask(),
                                                                  cm.getBlueMask());
                    BufferedImage rgbBuffer = new BufferedImage(newCM, newRaster, false,  null);
                    try { ImageIO.write(rgbBuffer, suffix, file); }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                else if (suffix.toLowerCase().equals("gif")) {
                    try { ImageIO.write(mixed, suffix, file); }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                else {
                    String msg = "Invalid image file type: " + suffix;
                    ImageIcon icon = createImageIcon("bad.png");
                    JOptionPane.showMessageDialog(frame, msg, "Invalid Suffix", JOptionPane.WARNING_MESSAGE, icon);
                }
                
            }
            else {
                ImageIcon icon = createImageIcon("bad.png");
                JOptionPane.showMessageDialog(frame, "Empty Filename", "Empty Filename", JOptionPane.WARNING_MESSAGE, icon);
            
        }
    }
        
    }
    
    private void makeScaledFromOrig() {
        
        int scaledwidth, scaledheight;
        
        if (original.width() >= original.height()) // landscape orientation
            scaledwidth = LONGSIDE;
        else scaledwidth = SHORTSIDE;
        
        scalefactor = (double)scaledwidth / (double)original.width();
        scaledheight = (int) (scalefactor*((double)original.height()));
        
        scaled = TransformationCentral.resize
            (original.getBuffedImage(), scaledwidth, scaledheight);
        
    }
    
    // this takes the current version of "scaled" and resizes it so 
    // the tiles fit perfectly
    private void makeScaledSnug() {
        
        int rows = js_row.getValue();
        int cols = js_col.getValue();
        
        // these two statements alters desired width so
        // it is a nice multiple of the tile width
        tilewidth = (int)((double)scaled.getWidth() / (double)cols);
        int scaledwidth = tilewidth*cols; // desiredwidth is updated
        
        scalefactor = (double)scaledwidth / (double)original.width();
        
        int scaledheight = (int)(scalefactor*original.height());
        tileheight = (int)((double)scaledheight / (double)rows);
        scaledheight = tileheight*rows; // desiredheight is updated
        
        //scaled image is updated so it has a nice fit
        scaled = TransformationCentral.resize(original.getBuffedImage(), scaledwidth, scaledheight);
        
    }
    
    /****************************************\
      JPanel for the displaying the main photo.
      \****************************************/
    class MainPhoto extends JPanel implements MouseListener, MouseMotionListener{
        
        public MainPhoto() {
            addMouseListener(this);
            addMouseMotionListener(this);
        }
        
        @Override public Dimension getPreferredSize(){
            if (scaled != null)
                return new Dimension(scaled.getWidth(), scaled.getHeight());
            else 
                return new Dimension(0,0);
        }
        
        private void drawGuides(Graphics2D g) {
            int tempwidth = scaled.getWidth()/js_col.getValue();
            int tempheight = scaled.getHeight()/js_row.getValue();
            
            if (guidecolor != null) {
                g.setPaint(guidecolor);
            }
                
            for (int x = 0; x < scaled.getWidth(); x += tempwidth)
                g.drawRect(x, 0, tempwidth, tempheight);
            for (int y = 0; y < scaled.getHeight(); y += tempheight)
                g.drawRect(0, y, tempwidth, tempheight);
        }
        
        // Methods, constructors, fields.
        @Override public void paintComponent(Graphics g) {
            super.paintComponent(g);    // paints background
            // do your drawing here
            
            Graphics2D g2 = (Graphics2D)g;
            
            if (scaled != null && transformswitch == false && showmosaic == false) {
                g2.drawImage(scaled, 0, 0, scaled.getWidth(), scaled.getHeight(), this);
                if (guides == true)
                    drawGuides(g2);
            }
            
            if (scaled != null && transformswitch == false && showmosaic == true) {
                g2.drawImage(mixed, 0, 0, mosaic.getWidth(), mosaic.getHeight(), this);
                if (guides == true)
                    drawGuides(g2);
            }
            
            if (transformswitch == true) {
                Graphics2D mosaicg = mosaic.createGraphics();
                mixed = new BufferedImage(mosaic.getWidth(), mosaic.getHeight(), BufferedImage.TYPE_INT_RGB);
                Graphics2D mixedg = mixed.createGraphics();
                boolean breaker = false; // needed for breaking both for-loops
                g2.drawImage(scaled, 0, 0, scaled.getWidth(), scaled.getHeight(), this);
                for (int row = 0; row < bestmatches.length; row++) {
                    for (int col = 0; col < bestmatches[0].length; col++) {
                        
                        // "draw" tile right here
                        if (bestmatches[row][col] == null) { breaker = true; } // once you hit a null cell, you can stop trying to paint
                        if (breaker == true) { break; }
                        else {
                            g2.drawImage(bestmatches[row][col].getBuffedImage(), col*tilewidth, row*tileheight, tilewidth, tileheight, this);
                            mosaicg.drawImage(bestmatches[row][col].getBuffedImage(), col*tilewidth, row*tileheight, tilewidth, tileheight, this);
                            mixedg.drawImage(bestmatches[row][col].getBuffedImage(), col*tilewidth, row*tileheight, tilewidth, tileheight, this);
                        }
                    }
                    if (breaker == true) { break; }
                }
                
            }
            
        }
        
        public void mouseMoved(MouseEvent e) {
            if (bestmatches != null) {
                
                //if (preview != null ) preview.dispose();
                
                int row = e.getY() / tileheight;
                int col = e.getX() / tilewidth;
                
                int GAP = 20; // moves image away from cursor
                
                preview.loadImg(bestmatches[row][col].getFilename());
                preview.setLocation(frame.getX() + e.getX() + GAP, frame.getY() + e.getY() + GAP);
                preview.setVisible(true);
                
            }
        }
        
        public void mouseDragged(MouseEvent e) {
            //eventOutput("Mouse dragged", e);
        }
        
        public void mousePressed(MouseEvent e) {
            //saySomething("Mouse pressed; # of clicks: "
            //                 + e.getClickCount(), e);
        }
        
        public void mouseReleased(MouseEvent e) {
            //saySomething("Mouse released; # of clicks: "
            //                 + e.getClickCount(), e);
        }
        
        public void mouseEntered(MouseEvent e) {
            if (bestmatches != null) {
                
                int row = e.getY() / tileheight;
                int col = e.getX() / tilewidth;
                
                int GAP = 20; // moves image away from cursor
                
                preview = new Preview(previewmin, previewmax);
                preview.loadImg(bestmatches[row][col].getFilename());
                preview.setLocation(frame.getX() + e.getX() + GAP, frame.getY() + e.getY() + GAP);
                preview.setVisible(true);
                
            }
        }
        
        public void mouseExited(MouseEvent e) {
            if (preview != null ) {
                preview.dispose();
                preview = null;
            }
        }
        
        public void mouseClicked(MouseEvent e) {
            //saySomething("Mouse clicked (# of clicks: "
            //                 + e.getClickCount() + ")", e);
        }
        
    }
    
    private static void centerWindow() {
        
        // Get the size of the screen
        Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
        
        // Determine the new location of the window
        int w = frame.getWidth();
        int h = frame.getHeight();
        int x = (dim.width-w)/2;
        int y = (dim.height-h)/2;
        
        // Move the window
        frame.setLocation(x, y);
        
    }
    
    // NOTE: this method assumes that image as already been
    // loaded to original but is not yet scaled into "mainphoto"
    private void reshapeWindow() {
        
        String title = frame.getTitle();
        frame.setTitle("");
        
        // RESIZING variables------------------------------------
        int SPACE = 70;
        int desiredwidth = scaled.getWidth() + jp_options_basic.getWidth() + SPACE;
        int desiredheight = scaled.getHeight() + SPACE;
        
        int STEPS = 12;
        
        // magnitude of width and height changes
        int magwdelta = Math.abs(desiredwidth - frame.getWidth()) / STEPS;
        int maghdelta = Math.abs(desiredheight - frame.getHeight()) / STEPS;
        
        if (frame.getWidth() < desiredwidth)
            magwdelta *= 1;
        else if (frame.getWidth() > desiredwidth)
            magwdelta *= -1;
        else if (frame.getWidth() == desiredwidth)
            magwdelta = 0;
        
        if (frame.getHeight() < desiredheight)
            maghdelta *= 1;
        else if (frame.getHeight() > desiredheight)
            maghdelta *= -1;
        else if (frame.getHeight() == desiredheight)
            maghdelta = 0;
        // END---------------------------------------------------
        
        // MOVEMENT variables---------------------------------------
        Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
        
        int startingx = frame.getX();
        int startingy = frame.getY();
        
        int desiredx = (dim.width - desiredwidth) / 2;
        int desiredy = (dim.height - desiredheight) / 2;
        
        int magxdelta = Math.abs(desiredx - startingx) / STEPS;
        int magydelta = Math.abs(desiredy - startingy) / STEPS;
        
        if (frame.getX() < desiredx)
            magxdelta *= 1;
        else if (frame.getX() > desiredx)
            magxdelta *= -1;
        else if (frame.getX() == desiredx)
            magxdelta = 0;
        
        if (frame.getY() < desiredy)
            magydelta *= 1;
        else if (frame.getY() > desiredy)
            magydelta *= -1;
        else if (frame.getY() == desiredy)
            magydelta = 0;
        // END------------------------------------------------------
        
        // animation for moving the window to a better location
        while (magwdelta != 0 && maghdelta != 0 
                   && magxdelta != 0 && magydelta != 0) {
            
            int newx = frame.getX() + magxdelta;
            int newy = frame.getY() + magydelta;
            int newwidth = frame.getWidth() + magwdelta;
            int newheight = frame.getHeight() + maghdelta;
            
            // If current attritubes overshoot or undershoot desired attributes-----
            if (newwidth > desiredwidth && magwdelta > 0) newwidth = desiredwidth;
            if (newwidth < desiredwidth && magwdelta < 0) newwidth = desiredwidth;
            if (newheight > desiredheight && maghdelta > 0) newheight = desiredheight;
            if (newheight < desiredheight && maghdelta < 0) newheight = desiredheight;
            
            if (newx > desiredx && magxdelta > 0) newx = desiredx;
            if (newx < desiredx && magxdelta < 0) newx = desiredx;
            if (newy > desiredy && magydelta > 0) newy = desiredy;
            if (newy < desiredy && magydelta < 0) newy = desiredy;
            // END------------------------------------------------------------------
            
            // Move and resize the window
            frame.setSize(newwidth, newheight);
            frame.setLocation(newx, newy);
            
            if (frame.getX() == desiredx)
                magxdelta = 0;
            if (frame.getY() == desiredy)
                magydelta = 0;
            if (frame.getWidth() == desiredwidth) 
                magwdelta = 0;
            if (frame.getHeight() == desiredheight) 
                maghdelta = 0;
        }
        
        frame.setTitle(title);
        
        //this is necessary for resolving display issues
        mainphoto.setBounds(mainphoto.getX(), mainphoto.getY(), scaled.getWidth(), scaled.getHeight());
    }
    
    public void delay(int t) {
        try { Thread.currentThread().sleep(t); }
        catch (InterruptedException e) { System.out.println("Error sleeping"); }
    }
    
    public void unlockTransformIfReady() {
        
        if (scaled != null && dbloaded == true) {
            bu_transform.setEnabled(true);
            bu_savehr.setEnabled(true);
        }
        
    }
    
    public BufferedImage createHiRes() {
        
        // hi res values must be numbers
        if (!okayHiResVals())
            return null;
        
        int rows = (int)Double.parseDouble(tf_rows.getText().trim());
        int cols = (int)Double.parseDouble(tf_cols.getText().trim());
        
        double inches = Double.parseDouble(tf_length.getText().trim());
        int width = convertToPixels(inches);
        double scalefactor = (double)width / (double)original.width();
        int height = (int)(scalefactor*((double)original.height()));
        
        double hrtilewidth = (int)((double)width / (double)cols);
        double hrtileheight = (int)((double)height / (double)rows);
        
        // rounding errors (dealing with ints and divisions)
        while (hrtileheight*rows < height)
            rows++;
        
        while (hrtilewidth*cols < width)
            cols++;
        
        double ortilewidth = ((double)original.width() / (double)cols);
        double ortileheight = ((double)original.height() / (double)rows);
        
        BufferedImage hires = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        Graphics2D hiresg = hires.createGraphics();
        
        int tiles = rows*cols;
        double delta  = 100.0 / tiles;
        double progress = 0.0;
        
        // mosaic must be done again because the 2d array "bestmatches" holds the resized version of each tile
        // not the original resolution (which may be needed for a higher resolution mosaic
        for (int row = 0; row < rows; row++) {
            for (int col = 0; col < cols; col++) {
                // these get adjusted if need by and preserve the variables
                // "tilewidth" and "tileheight"
                int ortempwidth = (int)ortilewidth;
                int ortempheight = (int)ortileheight;
                
                if ((row*ortileheight + ortileheight) > original.height())
                    ortempheight = (int)(original.height() - (double)row*ortileheight);
                if ((col*ortilewidth + ortilewidth) > original.width())
                    ortempwidth = (int)(original.width() - (double)col*ortilewidth);
                
                BufferedImage subset = original.getBuffedImage().getSubimage((int)(col*ortilewidth), (int)(row*ortileheight), ortempwidth, ortempheight);
                hiresg.drawImage(Comparer.getBestMatches(subset, tolerance).getBuffedImage(), (int)(col*hrtilewidth), (int)(row*hrtileheight), (int)hrtilewidth, (int)hrtileheight, this);
                progress += delta;
                pb_hires.setValue((int)progress);
                // difficult painting JUST the progress bar, so paint right half of frame
                this.paintImmediately(this.getX() + this.getWidth()/2, this.getY(), this.getWidth()/2, this.getHeight());
            }
        }
        pb_hires.setValue(100);
        
        return blender(hires, TransformationCentral.resize(original.getBuffedImage(), width, height));
        
    }
    
    private boolean okayHiResVals() {
        
        if (!InfoFinder.checkIfDouble(tf_compwidth.getText().trim())
                || !InfoFinder.checkIfDouble(tf_length.getText().trim())) {
            ImageIcon icon = createImageIcon("bad.png");
            JOptionPane.showMessageDialog(frame, "Values must be numbers.",
                                          "Invalid values", JOptionPane.WARNING_MESSAGE, icon);
            return false;
        }
        
        if (!InfoFinder.checkIfInt(tf_rows.getText().trim())
                || !InfoFinder.checkIfInt(tf_cols.getText().trim())) {
            ImageIcon icon = createImageIcon("bad.png");
            JOptionPane.showMessageDialog(frame, "Values must be integers.",
                                          "Invalid values", JOptionPane.WARNING_MESSAGE, icon);
            return false;
        }
        
        return true;
        
    }
    
    public void saveHiRes() {
        
        if (!okayHiResVals())
            return;
        
        FileDialog chooser = new FileDialog(frame, "Use a .png, .jpg, or .gif extension", FileDialog.SAVE);
        chooser.setVisible(true);
        String filename = chooser.getFile();
        
        jl_hallelujah.setVisible(false);
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        
        if (filename != null) {
            if( !filename.trim().equals("") ) {
                
                String fullpath = chooser.getDirectory() + File.separator + chooser.getFile();
                
                File file = new File(fullpath);
                String suffix = fullpath.substring(fullpath.lastIndexOf('.') + 1);
                
                // png files
                if (suffix.toLowerCase().equals("png")) {
                    try { 
                        BufferedImage hiresmosaic = createHiRes();
                        jl_hallelujah.setVisible(true);
                        ImageIO.write(hiresmosaic, suffix, file); 
                    }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                // need to change from ARGB to RGB for jpeg
                // reference: http://archives.java.sun.com/cgi-bin/wa?A2=ind0404&L=java2d-interest&D=0&P=2727
                else if (suffix.toLowerCase().equals("jpg") || suffix.toLowerCase().equals("jpeg")) {
                    
                    BufferedImage hiresmosaic = createHiRes();
                    jl_hallelujah.setVisible(true);
                    
                    WritableRaster raster = hiresmosaic.getRaster();
                    WritableRaster newRaster;
                    newRaster = raster.createWritableChild(0, 0, hiresmosaic.getWidth(), hiresmosaic.getHeight(), 0, 0, new int[] {0, 1, 2});
                    DirectColorModel cm = (DirectColorModel) hiresmosaic.getColorModel();
                    DirectColorModel newCM = new DirectColorModel(cm.getPixelSize(),
                                                                  cm.getRedMask(),
                                                                  cm.getGreenMask(),
                                                                  cm.getBlueMask());
                    BufferedImage rgbBuffer = new BufferedImage(newCM, newRaster, false,  null);
                    try { ImageIO.write(rgbBuffer, suffix, file); }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                else if (suffix.toLowerCase().equals("gif")) {
                    try { 
                        BufferedImage hiresmosaic = createHiRes();
                        jl_hallelujah.setVisible(true);
                        ImageIO.write(hiresmosaic, suffix, file); 
                    }
                    catch (IOException e) { e.printStackTrace(); }
                }
                
                else {
                    String msg = "Invalid image file type: " + suffix;
                    ImageIcon icon = createImageIcon("bad.png");
                    JOptionPane.showMessageDialog(frame, msg, "Invalid Suffix", JOptionPane.WARNING_MESSAGE, icon);
                }
                
            }
            else {
                ImageIcon icon = createImageIcon("bad.png");
                JOptionPane.showMessageDialog(frame,"Empty Filename",
                                              "Invalid filename",
                                              JOptionPane.WARNING_MESSAGE, icon);
            }
        }
        
        setCursor(null);
        
    }
    
    private int convertToPixels(double inches) {
        
        int widthres = Toolkit.getDefaultToolkit().getScreenSize().width; // width resolution
        int ppi = (int)((double)widthres / Double.parseDouble(tf_compwidth.getText().trim()));
        return (int)(inches*ppi);
        
    }
    
// prerequisite: both arguments must have same dimensions
    private BufferedImage blender(BufferedImage mosaicimg, BufferedImage normalimg){
        
        if (mosaicimg.getWidth() != normalimg.getWidth() || mosaicimg.getHeight() != normalimg.getHeight())
            throw new RuntimeException("Blended images must have equal dimensions.");
        
        BufferedImage blended = new BufferedImage(normalimg.getWidth(), normalimg.getHeight(), BufferedImage.TYPE_INT_RGB);
        
        for (int x = 0; x < normalimg.getWidth(); x++) {
            for (int y = 0; y < normalimg.getHeight(); y++) {
                
                Color normalpixel = new Color(normalimg.getRGB(x, y));
                Color mosaicpixel = new Color(mosaicimg.getRGB(x, y));
                
                // the slider's values range from 0 to 100
                double sliderfraction = (double)js_ghost.getValue() / 100.0;
                
                int newred = mosaicpixel.getRed() + (int)((double)(normalpixel.getRed() - mosaicpixel.getRed())*(sliderfraction));
                int newgreen = mosaicpixel.getGreen() + (int)((double)(normalpixel.getGreen() - mosaicpixel.getGreen())*(sliderfraction));
                int newblue = mosaicpixel.getBlue() + (int)((double)(normalpixel.getBlue() - mosaicpixel.getBlue())*(sliderfraction));
                
                Color newpixel = new Color(newred, newgreen, newblue);
                
                blended.setRGB(x, y, newpixel.getRGB());
            }
        }
        
        return blended;
        
    }
    
    public void resetGhost() { js_ghost.setValue(0); }
    
    /******\
      Client. 
      \******/
    public static void main(String[] args) {
        //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        javax.swing.SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                createAndShowGUI();
            }
        });
    }
    
}