/*************************************************************************
  *  Compilation:  javac Picture.java
  *  Execution:    java Picture imagename
  *
  *  Data type for manipulating individual pixels of an image. The original
  *  image can be read from a file in jpg, gif, or png format, or the
  *  user can create a blank image of a given size. Includes methods for
  *  displaying the image in a window on the screen or saving to a file.
  *
  *  % java Picture mandrill.jpg
  *
  *  Remarks
  *  -------
  *   - pixel (x, y) is column x and row y, where (0, 0) is upper left
  *
  *   - see also GrayPicture.java for a grayscale version
  * 
  * JOHN O'NEILL'S STREAMLINED VERSION
  *
  *************************************************************************/

import javax.imageio.ImageIO;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.awt.RenderingHints;
import java.awt.Graphics2D;
import java.io.File;
import java.io.IOException;
import java.net.URL;


/**
 *  This class provides methods for manipulating individual pixels of
 *  an image. The original image can be read from a file in JPEG, GIF,
 *  or PNG format, or the user can create a blank image of a given size.
 *  This class includes methods for displaying the image in a window on
 *  the screen or saving to a file.
 *  <p>
 *  Pixel (x, y) is column x, row y, where (0, 0) is upper left.
 *  <p>
 *  For additional documentation, see
 *  <a href="http://www.cs.princeton.edu/introcs/31datatype">Section 3.1</a> of
 *  <i>Introduction to Programming in Java: An Interdisciplinary Approach</i>
 *  by Robert Sedgewick and Kevin Wayne.
 */
public final class Picture {
    private BufferedImage image;    // the rasterized image
    private String filename;        // name of file
    
    /**
     * Create an empty w-by-h picture.
     */
    public Picture(int w, int h) {
        image = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
        // set to TYPE_INT_ARGB to support transparency
        filename = w + "-by-" + h;
    }
    
    /**
     * Create a picture by reading in a .png, .gif, or .jpg from
     * the given filename or URL name.
     */
    public Picture(String filename) {
        this.filename = filename;
        try {
            // try to read from file in working directory
            File file = new File(filename);
            if (file.isFile()) {
                image = ImageIO.read(file);
            }
            
            // now try to read from file in same directory as this .class file
            else {
                URL url = getClass().getResource(filename);
                if (url == null) { url = new URL(filename); }
                image = ImageIO.read(url);
            }
        } 
        catch (IOException e) {
            // e.printStackTrace();
            throw new RuntimeException("Could not open file: " + filename);
        }
        
        // check that image was read in
        if (image == null) {
            throw new RuntimeException("Invalid image file: " + filename);
        }
    }
    
    /**
     * Create a picture by reading in a .png, .gif, or .jpg from a File.
     */
    public Picture(File file) {
        try { image = ImageIO.read(file); }
        catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException("Could not open file: " + file);
        }
        if (image == null) {
            throw new RuntimeException("Invalid image file: " + file);
        }
    }
    
    /**
     * Return the height of the picture (in pixels).
     */
    public int height() {
        return image.getHeight(null);
    }
    
    /**
     * Return the width of the picture (in pixels).
     */
    public int width() {
        return image.getWidth(null);
    }
    
    /**
     * Return the Color of pixel (i, j).
     */
    public Color get(int i, int j) {
        return new Color(image.getRGB(i, j));
    }
    
    /**
     * Set the Color of pixel (i, j) to c.
     */
    public void set(int i, int j, Color c) {
        if (c == null) { throw new RuntimeException("can't set Color to null"); }
        image.setRGB(i, j, c.getRGB());
    }
    
    // CUSTOM
    public String getFilename() {return filename;}
    
    // CUSTOM
    public BufferedImage getBuffedImage() {return image;}
    
    // CUSTOM
    public void resize(int width, int height) {
        
        // Create new (blank) image of required (scaled) size
        
        BufferedImage scaledImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        
        // Paint scaled version of image to new image
        
        Graphics2D graphics2D = scaledImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(this.getBuffedImage(), 0, 0, width, height, null);
        
        // clean up
        
        graphics2D.dispose();
        
        this.image = scaledImage;
        
    }
    
}